// License: GPL. For details, see LICENSE file.
package org.openstreetmap.josm.data;

import static org.junit.Assert.assertEquals;

import org.junit.BeforeClass;
import org.junit.Test;
import org.openstreetmap.josm.data.validation.tests.SimilarNamedWays;

public class SimilarNamesTest {

    private static SimilarNamedWays similarity_;

    /**
     * Setup SimilarityRules.
     */
    @BeforeClass
    public static void init() {
        similarity_ = new SimilarNamedWays();
        similarity_.addRegExprRule("\\d+", "0");
        similarity_.addRegExprRule("\\d+(st|nd|rd|th)", "0st");
        similarity_.addSynonyms("east", "west", "north", "south");
        similarity_.addSynonyms("first", "second", "third");
    }

    @Test
    public void testSimilarNames() {
        checkSimilarity("same string", "Testname", "Testname", false);
        checkSimilarity("different case", "Testname", "TestName", true);
        checkSimilarity("typo", "Testname", "Testxame", true);
        checkSimilarity("missing char", "Testname", "Testame", true);
        checkSimilarity("additional char", "Testname", "Testxname", true);
        checkSimilarity("2 changes", "Testname", "Tostxname", true);
        checkSimilarity("3 changes", "Testname", "Tostxnam", false);

        // regular expression rule
        checkSimilarity("same number", "track 1", "track 1", false);
        checkSimilarity("different number", "track 1", "track 2", false);
        checkSimilarity("different number length", "track 9", "track 10", false);
        checkSimilarity("multiple numbers", "track 8 - 9", "track 10 - 11", false);

        checkSimilarity("1st and 2nd", "1st Street", "2nd Street", false);
        checkSimilarity("1st case", "1St Street", "1st Street", true);
        checkSimilarity("1st and 2nd case", "1St Street", "2nd Street", true);
        checkSimilarity("3rd and 4th", "2rd Street", "4th Street", false);

        // synonyms
        checkSimilarity("east and west", "East Foothill Drive", "West Foothill Drive", false);
        checkSimilarity("east and west case", "east Foothill Drive", "West Foothill Drive", true);
        checkSimilarity("first and second", "First Street", "Second Street", false);
        checkSimilarity("first and second case", "First Street", "second Street", true);
        checkSimilarity("first and second typo", "Forst Street", "Second Street", true);
        checkSimilarity("first and second typo2", "First Street", "Socond Street", true);
        checkSimilarity("first and second 2 changes", "First Street", "Soconds Street", true);
        checkSimilarity("first and second 3 changes", "First Street", "Soconds Stret", false);
    }

    private void checkSimilarity(String message, String name1, String name2, boolean expected) {
        boolean actual = similarity_.similaryName(name1, name2);
        assertEquals(message, expected, actual);

    }
}
