#!/bin/bash
set -e

VENV_DIR=".venv"
VARS_DIR="vars"

GREEN='\033[0;32m'
BLUE='\033[0;34m'
NC='\033[0m'

log() { echo -e "${BLUE}[INFO] $1${NC}"; }
success() { echo -e "${GREEN}[OK] $1${NC}"; }

log "Ensuring system dependencies..."
sudo apt-get update -qq
sudo apt-get install -y -qq python3 python3-venv python3-pip git sshpass > /dev/null

if [ ! -d "$VENV_DIR" ]; then
    log "Creating local virtual environment..."
    python3 -m venv "$VENV_DIR"
fi

log "Installing Ansible and collections..."
"$VENV_DIR/bin/pip" install --upgrade pip ansible --quiet
"$VENV_DIR/bin/ansible-galaxy" collection install community.postgresql community.general --force > /dev/null

log "Checking configuration files..."

# ensure vars directory exists
if [ ! -d "$VARS_DIR" ]; then
    mkdir -p "$VARS_DIR"
    log "Created '$VARS_DIR' directory."
fi

# define file mappings
declare -A FILES
FILES["hosts.ini.example"]="hosts.ini"
FILES["vars/main.yml.example"]="vars/main.yml"
FILES["vars/secrets.yml.example"]="vars/secrets.yml"

for example in "${!FILES[@]}"; do
    target="${FILES[$example]}"

    if [ ! -f "$target" ]; then
        if [ -f "$example" ]; then
            cp "$example" "$target"
            log "Created default '$target' from '$example'."

            # secure secrets file immediately
            if [[ "$target" == *"secrets.yml" ]]; then
                chmod 600 "$target"
            fi
        else
            log "Warning: Example file '$example' missing. Skipping."
        fi
    fi
done

success "Environment ready."
echo ""
echo "Next steps:"
echo "1. Edit 'hosts.ini' with your target server IP."
echo "2. Edit 'vars/main.yml' and 'vars/secrets.yml' with your settings."
echo "3. Activate environment: source $VENV_DIR/bin/activate"
echo "4. Run playbook: ansible-playbook -i hosts.ini install_weblate.yml -K"
