// License: GPL. For details, see LICENSE file.
package org.openstreetmap.josm.data.validation.tests;

import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.openstreetmap.josm.testutils.JOSMTestRules;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * Unit test of {@link ParkingLanesConditional}.
 */
class ParkingLanesConditionalTest {

    private final ParkingLanesConditional test = new ParkingLanesConditional();

    /**
     * Setup test
     */
    @RegisterExtension
    @SuppressFBWarnings(value = "URF_UNREAD_PUBLIC_OR_PROTECTED_FIELD")
    public JOSMTestRules rule = new JOSMTestRules().presets();

    /**
     * Setup test
     * @throws Exception if an error occurs
     */
    @BeforeEach
    public void setUp() throws Exception {
        test.initialize();
    }

    /**
     * Unit test of {@link ParkingLanesConditional#isKeyValid}.
     */
    @Test
    void testKeyValid() {
        assertTrue(test.isKeyValid("parking:condition:right:conditional"));
        assertTrue(test.isKeyValid("parking:condition:left:conditional"));
        assertTrue(test.isKeyValid("parking:condition:both:conditional"));
        assertTrue(test.isKeyValid("parking:condition:right:maxstay:conditional"));
        assertTrue(test.isKeyValid("parking:condition:left:maxstay:conditional"));
        assertTrue(test.isKeyValid("parking:condition:both:maxstay:conditional"));
        assertTrue(test.isKeyValid("parking:condition:both:hgv:conditional"));
        assertTrue(test.isKeyValid("parking:condition:both:hgv:maxstay:conditional"));
        assertFalse(test.isKeyValid("parking:lane:conditional"));
        assertFalse(test.isKeyValid("parking:condition:conditional:both"));
        assertFalse(test.isKeyValid("parking:condition:maxstay:conditional"));
    }

    /**
     * Unit test of {@link ParkingLanesConditional#isValueValid}.
     */
    @Test
    void testValueValid() {
        // Regular parking
        assertTrue(test.isValueValid("parking:condition:right:conditional", "ticket @ (Mo-Fr 06:00-19:00)"));
        assertTrue(test.isValueValid("parking:condition:right:conditional", "no_parking @ (Oct 1-Apr 30: We 10:00-15:00)"));
        assertFalse(test.isValueValid("parking:condition:right:conditional", " @ (06:00-19:00)"));
        assertFalse(test.isValueValid("parking:condition:right:conditional", "ticket (06:00-19:00)"));
        assertFalse(test.isValueValid("parking:condition:right:conditional", "ticket @ ()"));
        assertFalse(test.isValueValid("parking:condition:right:conditional", "ticket @ "));
        assertFalse(test.isValueValid("parking:condition:right:conditional", "ticket @ (06:00/19:00)"));
        // FIXME: Support this
        //assertTrue(test.isValueValid("parking:condition:right:conditional", "ticket \"Taxa 3\" @ (Mo-Fr 07:00-19:00, Sa 11:00-17:00); no_parking @ (Mo 00:00-06:00)"));
        assertTrue(test.isValueValid("parking:condition:right:conditional", "no_parking @ (Mo-Fr 08:00-20:00, Sa 08:00-16:00); free @ (Mo-Fr 08:00-20:00, Sa 08:00-16:00 AND residents)"));
        // Date parking
        assertTrue(test.isValueValid("parking:condition:left:conditional", "no_parking @ (Jan 1-15, Feb 1-15, Mar 1-15, Apr 1-15, May 1-15, Jun 1-15, Jul 1-15, Aug 1-15, Sep 1-15, Oct 1-15, Nov 1-15, Dec 1-15)"));
        assertTrue(test.isValueValid("parking:condition:right:conditional", "no_parking @ (Jan 16-31, Feb 16-29, Mar 16-31, Apr 16-30, May 16-31, Jun 16-30, Jul 16-31, Aug 16-31, Sep 16-30, Oct 16-31, Nov 16-30, Dec 16-31)"));
        // Not supported unless strict checking is on
        // assertFalse(test.isValueValid("parking:condition:left:conditional", "no_parking @ (Jan-Dec 1-15)"));
        assertTrue(test.isValueValid("parking:condition:left:conditional", "no_parking @ (Jan 1-31/2, Feb 1-29/2, Mar 1-31/2, Apr 1-30/2, May 1-31/2, Jun 1-30/2, Jul 1-31/2, Aug 1-31/2, Sep 1-30/2, Oct 1-31/2, Nov 1-30/2, Dec 1-31/2)"));
        // Not supported unless strict checking is on
        // assertFalse(test.isValueValid("parking:condition:left:conditional", "no_parking @ (Jan-Dec 1-31/2)"));
        assertTrue(test.isValueValid("parking:condition:left:conditional", "no_parking @ (week 1-53/2 Tu 08:00-10:00)"));
        // Maxstay
        // FIXME: Support maxstay
        // FIXME: Add deprecated values to mapcss file?
        assertTrue(test.isValueValid("parking:condition:right:maxstay:conditional", "2 hours @ (Mo-Fr 09:30-16:00, Sa 07:00-18:30)"));
        assertTrue(test.isValueValid("parking:condition:right:maxstay:conditional", "2 hours @ (Mo-Sa 08:00-21:00 AND free)"));
        assertTrue(test.isValueValid("parking:condition:right:maxstay:conditional", "30 minutes @ disc"));
        // Vehicles
        assertTrue(test.isValueValid("parking:condition:both:hgv:conditional", "no_parking @ (Sa-Su 00:00-24:00, Mo-Fr 00:00-08:00, Mo-Fr 18:00-24:00)"));
        assertTrue(test.isValueValid("parking:condition:right:bus:conditional", "disc @ (Mo-Fr 07:00-09:00, Mo-Fr 15:00-18:00)"));
    }
}
