// License: Apache 2.0. For details, see LICENSE file.
package org.openstreetmap.josm.plugins.wkt;

import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.data.osm.Relation;
import org.openstreetmap.josm.data.osm.RelationMember;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.gui.progress.NullProgressMonitor;
import org.openstreetmap.josm.gui.progress.ProgressMonitor;
import org.openstreetmap.josm.io.AbstractReader;
import org.openstreetmap.josm.io.IllegalDataException;
import org.openstreetmap.josm.tools.Logging;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Scanner;
import java.util.TreeMap;
import java.util.stream.Collectors;

import static org.openstreetmap.josm.tools.I18n.tr;

/**
 * Reader that reads WKT files.
 * Modified version of geojson reader. Thanks to the geojson authors.
 */
public class WktReader extends AbstractReader {

   WktReader() {
        // Restricts visibility
    }

    private void parse(String wkt) {
        String type = wkt.substring(0, wkt.indexOf('(')).toUpperCase().trim();
        String coordinates = wkt.substring(wkt.indexOf('(')+1,wkt.lastIndexOf(')'));
        
        switch (type) {
	        case "POINT":
	            parsePoint(coordinates);
	            break;
	        case "MULTIPOINT":
	            parseMultiPoint(coordinates);
	            break;
	        case "LINESTRING":
	            parseLineString(coordinates);
	            break;
	        case "MULTILINESTRING":
	            parseMultiLineString(coordinates);
	            break;
	        case "POLYGON":
	            parsePolygon(coordinates);
	            break;
	        case "MULTIPOLYGON":
	            parseMultiPolygon(coordinates);
	            break;
	        case "GEOMETRYCOLLECTION":
	            parseGeometryCollection(coordinates);
	            break;
	        default:
	            parseUnknown(wkt);
        }
    }

    private void parseGeometryCollection(String geometries) {
    	while (geometries.length() > 0)
    	{
    		int ind = getElementFromCollection(geometries);
    		String geometry = geometries.substring(0,ind);
    		while (ind<geometries.length() && !Character.isLetter(geometries.charAt(ind)))
    		{
    			ind++;
    		}
    		geometries = geometries.substring(ind);
    		parse(geometry);
    	}
    }

    private void parsePoint(final String coordinates) {
    	String[] coord = coordinates.split(" ");
        double lat = Double.parseDouble(coord[1]);
        double lon = Double.parseDouble(coord[0]);
        Node node = createNode(lat, lon);
    }

    private void parseMultiPoint(final String coordinates) {
    	String[] points = coordinates.split(",");
        for (String point : points) {
        	parsePoint(removeBrackets(point));
        }
    }

    private void parseLineString(final String coordinates) {
    	String[] points = coordinates.split(",");
        if (points.length == 0) {
            return;
        }
        createWay(coordinates, false);
    }

    private void parseMultiLineString(final String coordinates) {
    	String[] lines = coordinates.split("\\),");
        for (String line : lines) {
            parseLineString(removeBrackets(line));
        }
    }

    private void parsePolygon(final String coordinates) {
    	String[] parts = coordinates.split("\\),");
  
        final Relation multipolygon = new Relation();
        multipolygon.put("type", "multipolygon");
        createWay(removeBrackets(parts[0]), true)
            .ifPresent(way -> multipolygon.addMember(new RelationMember("outer", way)));
            	
        for (int i=1; i<parts.length; i++) {
            createWay(removeBrackets(parts[1]), true)
                .ifPresent(way -> multipolygon.addMember(new RelationMember("inner", way)));
        }
        getDataSet().addPrimitive(multipolygon);
    }
    
    private void parsePolygon(final String coordinates, final Relation multipolygon) {
    	String[] parts = coordinates.split("\\),");

        multipolygon.put("type", "multipolygon");
        createWay(removeBrackets(parts[0]), true)
            .ifPresent(way -> multipolygon.addMember(new RelationMember("outer", way)));
            	
        for (int i=1; i<parts.length; i++) {
            createWay(removeBrackets(parts[1]), true)
                .ifPresent(way -> multipolygon.addMember(new RelationMember("inner", way)));
        }
    }

    private void parseMultiPolygon(final String coordinates) {
    	String[] polygons = coordinates.split("\\)\\),");
    	final Relation multipolygon = new Relation();
        for (String polygon : polygons) {
            parsePolygon(polygon, multipolygon);
        }
        getDataSet().addPrimitive(multipolygon);
    }

    private Node createNode(final double lat, final double lon) {
        final Node node = new Node(new LatLon(lat, lon));
        getDataSet().addPrimitive(node);
        return node;
    }

    private Optional<Way> createWay(final String coordinates, final boolean autoClose) {
    	String[] points = coordinates.split(",");
        if (points.length == 0) {
            return Optional.empty();
        }
        
        List<LatLon> latlons = new ArrayList<LatLon>();
        for (String point : points)
        {
        	String[] coord = point.trim().split(" ");
            double lat = Double.parseDouble(coord[1]);
            double lon = Double.parseDouble(coord[0]);
            latlons.add(new LatLon(lat,lon));
        }

        final int size = latlons.size();
        final boolean doAutoclose;
        if (size > 1) {
            if (latlons.get(0).equals(latlons.get(size - 1))) {
                // Remove last coordinate, but later add first node to the end
                latlons.remove(size - 1);
                doAutoclose = true;
            } else {
                doAutoclose = autoClose;
            }
        } else {
            doAutoclose = false;
        }

        final Way way = new Way();
        way.setNodes(latlons.stream().map(Node::new).collect(Collectors.toList()));
        if (doAutoclose) {
            way.addNode(way.getNode(0));
        }

        way.getNodes().stream().distinct().forEach(it -> getDataSet().addPrimitive(it));
        getDataSet().addPrimitive(way);

        return Optional.of(way);
    }
    
    private String removeBrackets(final String s)
    {
    	int start = 0;
    	for (int i = 0; i<s.length(); i++)
    	{
    		if (s.charAt(i) == '(') start = i+1;
    		else if (s.charAt(i) == ')') return s.substring(start,i).trim();
    	}
    	return s.substring(start,s.length()).trim();
    }
    
    private int getElementFromCollection(final String s)
    {
    	int ind = s.indexOf('(') + 1;
    	if (ind == 0) return -1;
    	int level = 1;
    	while (ind < s.length() && level > 0)
    	{
    		if (s.charAt(ind) == '(') level++;
    		else if (s.charAt(ind) == ')') level--;
    		ind++;
    	}
    	return ind;
    }

    private void parseUnknown(final String object) {
        Logging.warn(tr("Unknown wkt object found {0}", object));
    }

    @Override
    protected DataSet doParseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
    	
    	Scanner s = new Scanner(source).useDelimiter("\\A");
    	String text = s.hasNext() ? s.next() : "";
    	parse(text);

        return getDataSet();
    }

    /**
     * Parse the given input source and return the dataset.
     *
     * @param source          the source input stream. Must not be null.
     * @param progressMonitor the progress monitor. If null, {@link NullProgressMonitor#INSTANCE} is assumed
     * @return the dataset with the parsed data
     * @throws IllegalDataException     if an error was found while parsing the data from the source
     * @throws IllegalArgumentException if source is null
     */
    public static DataSet parseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        return new WktReader().doParseDataSet(source, progressMonitor);
    }

}
