/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.plugins.geojson;

import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonString;
import javax.json.JsonStructure;
import javax.json.JsonValue;
import javax.json.stream.JsonParser;
import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.data.osm.Relation;
import org.openstreetmap.josm.data.osm.RelationMember;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.gui.progress.ProgressMonitor;
import org.openstreetmap.josm.io.AbstractReader;
import org.openstreetmap.josm.io.IllegalDataException;
import org.openstreetmap.josm.tools.I18n;
import org.openstreetmap.josm.tools.Logging;

public class GeoJsonReader
extends AbstractReader {
    private static final String COORDINATES = "coordinates";
    private static final String FEATURES = "features";
    private static final String PROPERTIES = "properties";
    private static final String GEOMETRY = "geometry";
    private static final String TYPE = "type";
    private JsonParser parser;

    GeoJsonReader() {
    }

    private void setParser(JsonParser parser) {
        this.parser = parser;
    }

    private void parse() {
        while (this.parser.hasNext()) {
            JsonParser.Event event = this.parser.next();
            if (event != JsonParser.Event.START_OBJECT) continue;
            this.parseRoot(this.parser.getObject());
        }
        this.parser.close();
    }

    private void parseRoot(JsonObject object) {
        switch (object.getString(TYPE)) {
            case "FeatureCollection": {
                this.parseFeatureCollection(object.getJsonArray(FEATURES));
                break;
            }
            case "Feature": {
                this.parseFeature(object);
                break;
            }
            case "GeometryCollection": {
                this.parseGeometryCollection(null, object);
                break;
            }
            default: {
                this.parseGeometry(null, object);
            }
        }
    }

    private void parseFeatureCollection(JsonArray features) {
        for (JsonValue feature : features) {
            if (!(feature instanceof JsonObject)) continue;
            JsonObject item = (JsonObject)feature;
            this.parseFeature(item);
        }
    }

    private void parseFeature(JsonObject feature) {
        JsonValue geometry = (JsonValue)feature.get((Object)GEOMETRY);
        if (geometry != null && geometry.getValueType().equals((Object)JsonValue.ValueType.OBJECT)) {
            this.parseGeometry(feature, geometry.asJsonObject());
        } else {
            JsonValue properties = (JsonValue)feature.get((Object)PROPERTIES);
            if (properties != null && properties.getValueType().equals((Object)JsonValue.ValueType.OBJECT)) {
                this.parseNonGeometryFeature(feature, properties.asJsonObject());
            } else {
                Logging.warn((String)I18n.tr((String)"Relation/non-geometry feature without properties found: {0}", (Object[])new Object[]{feature}));
            }
        }
    }

    private void parseNonGeometryFeature(JsonObject feature, JsonObject properties) {
        JsonValue type = (JsonValue)properties.get((Object)TYPE);
        if (type == null || properties.getValueType().equals((Object)JsonValue.ValueType.STRING)) {
            Logging.warn((String)I18n.tr((String)"Relation/non-geometry feature without type found: {0}", (Object[])new Object[]{feature}));
            return;
        }
        Relation relation = new Relation();
        relation.put(TYPE, type.toString());
        this.fillTagsFromFeature(feature, (OsmPrimitive)relation);
        this.getDataSet().addPrimitive((OsmPrimitive)relation);
    }

    private void parseGeometryCollection(JsonObject feature, JsonObject geometry) {
        JsonArray geometries = geometry.getJsonArray("geometries");
        for (JsonValue jsonValue : geometries) {
            this.parseGeometry(feature, jsonValue.asJsonObject());
        }
    }

    private void parseGeometry(JsonObject feature, JsonObject geometry) {
        if (geometry == null) {
            this.parseNullGeometry(feature);
            return;
        }
        switch (geometry.getString(TYPE)) {
            case "Point": {
                this.parsePoint(feature, geometry.getJsonArray(COORDINATES));
                break;
            }
            case "MultiPoint": {
                this.parseMultiPoint(feature, geometry);
                break;
            }
            case "LineString": {
                this.parseLineString(feature, geometry.getJsonArray(COORDINATES));
                break;
            }
            case "MultiLineString": {
                this.parseMultiLineString(feature, geometry);
                break;
            }
            case "Polygon": {
                this.parsePolygon(feature, geometry.getJsonArray(COORDINATES));
                break;
            }
            case "MultiPolygon": {
                this.parseMultiPolygon(feature, geometry);
                break;
            }
            case "GeometryCollection": {
                this.parseGeometryCollection(feature, geometry);
                break;
            }
            default: {
                this.parseUnknown(geometry);
            }
        }
    }

    private void parsePoint(JsonObject feature, JsonArray coordinates) {
        double lat = coordinates.getJsonNumber(1).doubleValue();
        double lon = coordinates.getJsonNumber(0).doubleValue();
        Node node = this.createNode(lat, lon);
        this.fillTagsFromFeature(feature, (OsmPrimitive)node);
    }

    private void parseMultiPoint(JsonObject feature, JsonObject geometry) {
        JsonArray coordinates = geometry.getJsonArray(COORDINATES);
        for (JsonValue coordinate : coordinates) {
            this.parsePoint(feature, coordinate.asJsonArray());
        }
    }

    private void parseLineString(JsonObject feature, JsonArray coordinates) {
        if (coordinates.isEmpty()) {
            return;
        }
        this.createWay(coordinates, false).ifPresent(way -> this.fillTagsFromFeature(feature, (OsmPrimitive)way));
    }

    private void parseMultiLineString(JsonObject feature, JsonObject geometry) {
        JsonArray coordinates = geometry.getJsonArray(COORDINATES);
        for (JsonValue coordinate : coordinates) {
            this.parseLineString(feature, coordinate.asJsonArray());
        }
    }

    private void parsePolygon(JsonObject feature, JsonArray coordinates) {
        if (coordinates.size() == 1) {
            this.createWay(coordinates.getJsonArray(0), true).ifPresent(way -> this.fillTagsFromFeature(feature, (OsmPrimitive)way));
        } else if (coordinates.size() > 1) {
            Relation multipolygon = new Relation();
            multipolygon.put(TYPE, "multipolygon");
            this.createWay(coordinates.getJsonArray(0), true).ifPresent(way -> multipolygon.addMember(new RelationMember("outer", (OsmPrimitive)way)));
            for (JsonValue interiorRing : coordinates.subList(1, coordinates.size())) {
                this.createWay(interiorRing.asJsonArray(), true).ifPresent(way -> multipolygon.addMember(new RelationMember("inner", (OsmPrimitive)way)));
            }
            this.fillTagsFromFeature(feature, (OsmPrimitive)multipolygon);
            this.getDataSet().addPrimitive((OsmPrimitive)multipolygon);
        }
    }

    private void parseMultiPolygon(JsonObject feature, JsonObject geometry) {
        JsonArray coordinates = geometry.getJsonArray(COORDINATES);
        for (JsonValue coordinate : coordinates) {
            this.parsePolygon(feature, coordinate.asJsonArray());
        }
    }

    private Node createNode(double lat, double lon) {
        Node node = new Node(new LatLon(lat, lon));
        this.getDataSet().addPrimitive((OsmPrimitive)node);
        return node;
    }

    private Optional<Way> createWay(JsonArray coordinates, boolean autoClose) {
        boolean doAutoclose;
        if (coordinates.isEmpty()) {
            return Optional.empty();
        }
        List latlons = coordinates.stream().map(coordinate -> {
            JsonArray jsonValues = coordinate.asJsonArray();
            return new LatLon(jsonValues.getJsonNumber(1).doubleValue(), jsonValues.getJsonNumber(0).doubleValue());
        }).collect(Collectors.toList());
        int size = latlons.size();
        if (size > 1) {
            if (((LatLon)latlons.get(0)).equals(latlons.get(size - 1))) {
                latlons.remove(size - 1);
                doAutoclose = true;
            } else {
                doAutoclose = autoClose;
            }
        } else {
            doAutoclose = false;
        }
        Way way = new Way();
        way.setNodes(latlons.stream().map(Node::new).collect(Collectors.toList()));
        if (doAutoclose) {
            way.addNode(way.getNode(0));
        }
        way.getNodes().stream().distinct().forEach(it -> this.getDataSet().addPrimitive((OsmPrimitive)it));
        this.getDataSet().addPrimitive((OsmPrimitive)way);
        return Optional.of(way);
    }

    private void fillTagsFromFeature(JsonObject feature, OsmPrimitive primitive) {
        if (feature != null) {
            primitive.setKeys(this.getTags(feature));
        }
    }

    private void parseUnknown(JsonObject object) {
        Logging.warn((String)I18n.tr((String)"Unknown json object found {0}", (Object[])new Object[]{object}));
    }

    private void parseNullGeometry(JsonObject feature) {
        Logging.warn((String)I18n.tr((String)"Geometry of feature {0} is null", (Object[])new Object[]{feature}));
    }

    private Map<String, String> getTags(JsonObject feature) {
        JsonValue properties;
        TreeMap<String, String> tags = new TreeMap<String, String>();
        if (feature.containsKey((Object)PROPERTIES) && !feature.isNull(PROPERTIES) && (properties = (JsonValue)feature.get((Object)PROPERTIES)) != null && properties.getValueType().equals((Object)JsonValue.ValueType.OBJECT)) {
            for (Map.Entry stringJsonValueEntry : properties.asJsonObject().entrySet()) {
                JsonValue value = (JsonValue)stringJsonValueEntry.getValue();
                if (value instanceof JsonString) {
                    tags.put((String)stringJsonValueEntry.getKey(), ((JsonString)value).getString());
                    continue;
                }
                if (value instanceof JsonStructure) {
                    Logging.warn((String)("The GeoJSON contains an object with property '" + (String)stringJsonValueEntry.getKey() + "' whose value has the unsupported type '" + value.getClass().getSimpleName() + "'. That key-value pair is ignored!"));
                    continue;
                }
                if (value.getValueType() == JsonValue.ValueType.NULL) {
                    tags.put((String)stringJsonValueEntry.getKey(), null);
                    continue;
                }
                tags.put((String)stringJsonValueEntry.getKey(), value.toString());
            }
        }
        return tags;
    }

    protected DataSet doParseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        this.setParser(Json.createParser((InputStream)source));
        this.parse();
        return this.getDataSet();
    }

    public static DataSet parseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        return new GeoJsonReader().doParseDataSet(source, progressMonitor);
    }
}

