// License: GPL. For details, see LICENSE file.
package org.openstreetmap.josm.data;

import static org.openstreetmap.josm.tools.I18n.tr;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.regex.Pattern;

import org.openstreetmap.josm.Main;
import org.openstreetmap.josm.actions.OpenFileAction.OpenFileTask;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.event.AbstractDatasetChangedEvent;
import org.openstreetmap.josm.data.osm.event.DataSetListenerAdapter;
import org.openstreetmap.josm.data.osm.event.DataSetListenerAdapter.Listener;
import org.openstreetmap.josm.data.preferences.BooleanProperty;
import org.openstreetmap.josm.data.preferences.IntegerProperty;
import org.openstreetmap.josm.gui.MapView;
import org.openstreetmap.josm.gui.MapView.LayerChangeListener;
import org.openstreetmap.josm.gui.layer.Layer;
import org.openstreetmap.josm.gui.layer.OsmDataLayer;
import org.openstreetmap.josm.io.OsmExporter;

/**
 * Saves data layers periodically so they can be recovered in case of a crash.
 */
public class AutosaveTask extends TimerTask implements LayerChangeListener, Listener {

    private static final char[] ILLEGAL_CHARACTERS = { '/', '\n', '\r', '\t', '\0', '\f', '`', '?', '*', '\\', '<', '>', '|', '\"', ':' };
    private static final String AUTOSAVE_DIR = "autosave";
    private static final String DELETED_LAYERS_DIR = "autosave/deleted_layers";

    public static final BooleanProperty PROP_AUTOSAVE_ENABLED = new BooleanProperty("autosave.enabled", true);
    public static final IntegerProperty PROP_FILES_PER_LAYER = new IntegerProperty("autosave.filesPerLayer", 1);
    public static final IntegerProperty PROP_DELETED_LAYERS = new IntegerProperty("autosave.deletedLayersBackupCount", 5);
    public static final IntegerProperty PROP_INTERVAL = new IntegerProperty("autosave.interval", 5 * 60);
    public static final IntegerProperty PROP_INDEX_LIMIT = new IntegerProperty("autosave.index-limit", 1000);

    private static class AutosaveLayerInfo {
        OsmDataLayer layer;
        String layerName;
        String layerFileName;
        final Deque<File> backupFiles = new LinkedList<File>();
    }

    private final DataSetListenerAdapter datasetAdapter = new DataSetListenerAdapter(this);
    private Set<DataSet> changedDatasets = new HashSet<DataSet>();
    private final List<AutosaveLayerInfo> layersInfo = new ArrayList<AutosaveLayerInfo>();
    private Timer timer;
    private final Object layersLock = new Object();
    private final Deque<File> deletedLayers = new LinkedList<File>();

    private final File autosaveDir = new File(Main.pref.getPreferencesDir() + AUTOSAVE_DIR);
    private final File deletedLayersDir = new File(Main.pref.getPreferencesDir() + DELETED_LAYERS_DIR);

    public void schedule() {
        if (PROP_INTERVAL.get() > 0) {

            if (!autosaveDir.exists() && !autosaveDir.mkdirs()) {
                System.out.println(tr("Unable to create directory {0}, autosave will be disabled", autosaveDir.getAbsolutePath()));
                return;
            }
            if (!deletedLayersDir.exists() && !deletedLayersDir.mkdirs()) {
                System.out.println(tr("Unable to create directory {0}, autosave will be disabled", deletedLayersDir.getAbsolutePath()));
                return;
            }

            for (File f: deletedLayersDir.listFiles()) {
                deletedLayers.add(f);
            }


            timer = new Timer(true);
            timer.schedule(this, 1000, PROP_INTERVAL.get() * 1000);
            MapView.addLayerChangeListener(this);
            if (Main.isDisplayingMapView()) {
                for (OsmDataLayer l: Main.map.mapView.getLayersOfType(OsmDataLayer.class)) {
                    registerNewlayer(l);
                }
            }
        }
    }

    private String getFileName(String layerName, int index) {
        String result = layerName;
        for (int i=0; i<ILLEGAL_CHARACTERS.length; i++) {
            result = result.replaceAll(Pattern.quote(String.valueOf(ILLEGAL_CHARACTERS[i])),
                    '&' + String.valueOf((int)ILLEGAL_CHARACTERS[i]) + ';');
        }
        if (index != 0) {
            result = result + '_' + index;
        }
        return result;
    }

    private void setLayerFileName(AutosaveLayerInfo layer) {
        int index = 0;
        while (true) {
            String filename = getFileName(layer.layer.getName(), index);
            boolean foundTheSame = false;
            for (AutosaveLayerInfo info: layersInfo) {
                if (info != layer && filename.equals(info.layerFileName)) {
                    foundTheSame = true;
                    break;
                }
            }

            if (!foundTheSame) {
                layer.layerFileName = filename;
                return;
            }

            index++;
        }
    }

    private File getNewLayerFile(AutosaveLayerInfo layer) {
        int index = 0;
        Date now = new Date();
        while (true) {
            File result = new File(autosaveDir, String.format("%1$s_%2$tY%2$tm%2$td_%2$tH%2$tM%3$s.osm", layer.layerFileName, now, index == 0?"":"_" + index));
            try {
                if (result.createNewFile())
                    return result;
                else {
                    System.out.println(tr("Unable to create file {0}, other filename will be used", result.getAbsolutePath()));
                    if (index > PROP_INDEX_LIMIT.get())
                        throw new IOException("index limit exceeded");
                }
            } catch (IOException e) {
                System.err.println(tr("IOError while creating file, autosave will be skipped: {0}", e.getMessage()));
                return null;
            }
            index++;
        }
    }

    private void savelayer(AutosaveLayerInfo info) throws IOException {
        if (!info.layer.getName().equals(info.layerName)) {
            setLayerFileName(info);
            info.layerName = info.layer.getName();
        }
        if (changedDatasets.contains(info.layer.data)) {
            File file = getNewLayerFile(info);
            if (file != null) {
                info.backupFiles.add(file);
                new OsmExporter().exportData(file, info.layer);
            }
        }
        while (info.backupFiles.size() > PROP_FILES_PER_LAYER.get()) {
            File oldFile = info.backupFiles.remove();
            if (!oldFile.delete()) {
                System.out.println(tr("Unable to delete old backup file {0}", oldFile.getAbsolutePath()));
            }
        }
    }

    @Override
    public void run() {
        synchronized (layersLock) {
            try {
                for (AutosaveLayerInfo info: layersInfo) {
                    savelayer(info);
                }
                changedDatasets.clear();
            } catch (Throwable t) {
                // Don't let exception stop time thread
                System.err.println("Autosave failed: ");
                t.printStackTrace();
            }
        }
    }

    @Override
    public void activeLayerChange(Layer oldLayer, Layer newLayer) {
        // Do nothing
    }

    private void registerNewlayer(OsmDataLayer layer) {
        synchronized (layersLock) {
            layer.data.addDataSetListener(datasetAdapter);
            AutosaveLayerInfo info = new AutosaveLayerInfo();
            info.layer = layer;
            layersInfo.add(info);
        }
    }

    @Override
    public void layerAdded(Layer newLayer) {
        if (newLayer instanceof OsmDataLayer) {
            registerNewlayer((OsmDataLayer) newLayer);
        }
    }

    @Override
    public void layerRemoved(Layer oldLayer) {
        if (oldLayer instanceof OsmDataLayer) {
            synchronized (layersLock) {
                OsmDataLayer osmLayer = (OsmDataLayer) oldLayer;
                osmLayer.data.removeDataSetListener(datasetAdapter);
                Iterator<AutosaveLayerInfo> it = layersInfo.iterator();
                while (it.hasNext()) {
                    AutosaveLayerInfo info = it.next();
                    if (info.layer == osmLayer) {

                        try {
                            savelayer(info);
                            File lastFile = info.backupFiles.pollLast();
                            if (lastFile != null) {
                                moveToDeletedLayersFolder(lastFile);
                            }
                            for (File file: info.backupFiles) {
                                file.delete();
                            }
                        } catch (IOException e) {
                            System.err.println(tr("Error while creating backup of removed layer: {0}", e.getMessage()));
                        }

                        it.remove();
                    }
                }
            }
        }
    }

    @Override
    public void processDatasetEvent(AbstractDatasetChangedEvent event) {
        changedDatasets.add(event.getDataset());
    }

    public List<File> getUnsavedLayersFiles() {
        List<File> result = new ArrayList<File>();
        File[] files = autosaveDir.listFiles();
        System.err.println("autosave debug (getUnsavedLayersFiles) files="+(files == null ? null : Arrays.toString(files)));
        if (files == null)
            return result;
        for (File file: files) {
            System.err.println("autosave debug (getUnsavedLayersFiles) file="+file);

            if (file.isFile()) {
                System.err.println("autosave debug (getUnsavedLayersFiles) isFile");
                result.add(file);
            }
        }
        System.err.println("autosave debug (getUnsavedLayersFiles) result="+result);
        return result;
    }

    public void recoverUnsavedLayers() {
        List<File> files = getUnsavedLayersFiles();
        final OpenFileTask openFileTsk = new OpenFileTask(files, null, tr("Restoring files"));
        Main.worker.submit(openFileTsk);
        Main.worker.submit(new Runnable() {
            public void run() {
                for (File f: openFileTsk.getSuccessfullyOpenedFiles()) {
                    moveToDeletedLayersFolder(f);
                }
            }          
        });
    }

    private void moveToDeletedLayersFolder(File f) {
        System.err.println("autosave debug (moveToDeletedLayersFolder) f="+f);
        System.err.println("autosave debug (moveToDeletedLayersFolder) f.getName="+f.getName());

        File backupFile = new File(deletedLayersDir, f.getName());

        System.err.println("autosave debug (moveToDeletedLayersFolder) backupFile="+backupFile);
        if (backupFile.exists()) {
            System.err.println("autosave debug (moveToDeletedLayersFolder) backupFile exisist");

            boolean res = deletedLayers.remove(backupFile);

            System.err.println("autosave debug (moveToDeletedLayersFolder) res="+res);

            boolean res2 = backupFile.delete();

            System.err.println("autosave debug (moveToDeletedLayersFolder) res2="+res2);
        }
        if (f.renameTo(backupFile)) {
            System.err.println("autosave debug (moveToDeletedLayersFolder) rename ok");

            deletedLayers.add(backupFile);

            System.err.println("autosave debug (moveToDeletedLayersFolder) deletedLayers="+deletedLayers);
        } else {
            System.err.println(String.format("Warning: Could not move autosaved file %s to %s folder", f.getName(), deletedLayersDir.getName()));
            boolean res3 = f.delete();

            System.err.println("autosave debug (moveToDeletedLayersFolder) res3="+res3);
        }
        while (deletedLayers.size() > PROP_DELETED_LAYERS.get()) {
            File next = deletedLayers.remove();

            System.err.println("autosave debug (moveToDeletedLayersFolder) next="+next);

            boolean res4 = next.delete();

            System.err.println("autosave debug (moveToDeletedLayersFolder) res4="+res4);
        }
    }

    public void dicardUnsavedLayers() {
        List<File> ulfs = getUnsavedLayersFiles();
        System.err.println("autosave debug (dicardUnsavedLayers) ulfs="+ulfs);
        for (File f: ulfs) {
            System.err.println("autosave debug (dicardUnsavedLayers) f="+f);
            moveToDeletedLayersFolder(f);
        }
    }
}
