/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.plugins.elevation;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.openstreetmap.josm.data.Bounds;
import org.openstreetmap.josm.data.Preferences;
import org.openstreetmap.josm.data.coor.ILatLon;
import org.openstreetmap.josm.io.Compression;
import org.openstreetmap.josm.tools.CheckParameterUtil;
import org.openstreetmap.josm.tools.Logging;

public class HgtReader {
    private static final int SRTM_EXTENT = 1;
    private static final List<String> COMPRESSION_EXT = Arrays.asList("xz", "gzip", "zip", "bz", "bz2");
    public static final String HGT_EXT = ".hgt";
    public static final int HGT_VOID = Short.MIN_VALUE;
    private static final HashMap<String, short[][]> cache = new HashMap();

    public static double getElevationFromHgt(ILatLon coor) {
        try {
            String file = HgtReader.getHgtFileName(coor);
            if (!cache.containsKey(file)) {
                cache.put(file, null);
                for (String location : Preferences.getAllPossiblePreferenceDirs()) {
                    String fullPath = new File(location + File.separator + "elevation", file).getPath();
                    File f = new File(fullPath);
                    if (!f.exists()) {
                        String ext;
                        Iterator<String> iterator = COMPRESSION_EXT.iterator();
                        while (iterator.hasNext() && !(f = new File(fullPath + "." + (ext = iterator.next()))).exists()) {
                        }
                    }
                    if (!f.exists()) continue;
                    HgtReader.read(f);
                    break;
                }
            }
            return HgtReader.readElevation(coor, file);
        }
        catch (FileNotFoundException e) {
            Logging.error((String)("Get elevation from HGT " + String.valueOf(coor) + " failed: => " + e.getMessage()));
            return Double.NaN;
        }
        catch (Exception ioe) {
            Logging.error((Throwable)ioe);
            return Double.NaN;
        }
    }

    public static Bounds read(File file) throws IOException {
        String location = file.getName();
        for (String ext : COMPRESSION_EXT) {
            location = location.replaceAll("\\." + ext + "$", "");
        }
        short[][] sb = HgtReader.readHgtFile(file.getPath());
        cache.put(location, sb);
        Pattern pattern = Pattern.compile("([NS])(\\d{2})([EW])(\\d{3})");
        Matcher matcher = pattern.matcher(location);
        if (matcher.lookingAt()) {
            int lat = ("S".equals(matcher.group(1)) ? -1 : 1) * Integer.parseInt(matcher.group(2));
            int lon = ("W".equals(matcher.group(3)) ? -1 : 1) * Integer.parseInt(matcher.group(4));
            return new Bounds((double)lat, (double)lon, (double)(lat + 1), (double)(lon + 1));
        }
        return null;
    }

    private static short[][] readHgtFile(String file) throws IOException {
        CheckParameterUtil.ensureParameterNotNull((Object)file);
        short[][] data = null;
        try (InputStream fis = Compression.getUncompressedFileInputStream((Path)Paths.get(file, new String[0]));){
            ByteBuffer bb = ByteBuffer.wrap(fis.readAllBytes());
            bb.order(ByteOrder.BIG_ENDIAN);
            int size = (int)Math.sqrt((double)bb.array().length / 2.0);
            data = new short[size][size];
            int y = 0;
            for (int x = 0; x < size; ++x) {
                while (y < size) {
                    data[x][y] = bb.getShort(2 * (x * size + y));
                    ++y;
                }
                y = 0;
            }
        }
        return data;
    }

    public static double readElevation(ILatLon coor) {
        String tag = HgtReader.getHgtFileName(coor);
        return HgtReader.readElevation(coor, tag);
    }

    public static double readElevation(ILatLon coor, String fileName) {
        short[][] sb = cache.get(fileName);
        if (sb == null) {
            return Double.NaN;
        }
        int[] index = HgtReader.getIndex(coor, sb.length);
        short ele = sb[index[0]][index[1]];
        if (ele == Short.MIN_VALUE) {
            return Double.NaN;
        }
        return ele;
    }

    public static Optional<Bounds> getBounds(ILatLon location) {
        String fileName = HgtReader.getHgtFileName(location);
        short[][] sb = cache.get(fileName);
        if (sb == null) {
            return Optional.empty();
        }
        double latDegrees = location.lat();
        double lonDegrees = location.lon();
        float fraction = 1.0f / (float)sb.length;
        int latitude = (int)Math.floor(latDegrees) + (latDegrees < 0.0 ? 1 : 0);
        int longitude = (int)Math.floor(lonDegrees) + (lonDegrees < 0.0 ? 1 : 0);
        int[] index = HgtReader.getIndex(location, sb.length);
        int latSign = latitude > 0 ? 1 : -1;
        int lonSign = longitude > 0 ? 1 : -1;
        double minLat = (float)latitude + (float)latSign * fraction * (float)index[0];
        double maxLat = (float)latitude + (float)latSign * fraction * (float)(index[0] + 1);
        double minLon = (float)longitude + (float)lonSign * fraction * (float)index[1];
        double maxLon = (float)longitude + (float)lonSign * fraction * (float)(index[1] + 1);
        return Optional.of(new Bounds(Math.min(minLat, maxLat), Math.min(minLon, maxLon), Math.max(minLat, maxLat), Math.max(minLon, maxLon)));
    }

    private static int[] getIndex(ILatLon latLon, int mapSize) {
        double latDegrees = latLon.lat();
        double lonDegrees = latLon.lon();
        float fraction = 1.0f / (float)(mapSize - 1);
        int latitude = (int)Math.round(HgtReader.frac(Math.abs(latDegrees)) / (double)fraction);
        int longitude = (int)Math.round(HgtReader.frac(Math.abs(lonDegrees)) / (double)fraction);
        if (latDegrees >= 0.0) {
            latitude = mapSize - latitude - 1;
        }
        if (lonDegrees < 0.0) {
            longitude = mapSize - longitude - 1;
        }
        return new int[]{latitude, longitude};
    }

    public static String getHgtFileName(ILatLon latLon) {
        int lat = (int)Math.floor(latLon.lat());
        int lon = (int)Math.floor(latLon.lon());
        String latPref = "N";
        if (lat < 0) {
            latPref = "S";
            lat = Math.abs(lat);
        }
        String lonPref = "E";
        if (lon < 0) {
            lonPref = "W";
            lon = Math.abs(lon);
        }
        return String.format("%s%2d%s%03d.hgt", latPref, lat, lonPref, lon);
    }

    public static double frac(double d) {
        long iPart = (long)d;
        double fPart = d - (double)iPart;
        return fPart;
    }

    public static void clearCache() {
        cache.clear();
    }
}

