/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.plugins.fit.lib;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.openstreetmap.josm.plugins.fit.lib.FitException;
import org.openstreetmap.josm.plugins.fit.lib.FitReaderOptions;
import org.openstreetmap.josm.plugins.fit.lib.global.FitData;
import org.openstreetmap.josm.plugins.fit.lib.global.Global;
import org.openstreetmap.josm.plugins.fit.lib.global.IFitTimestamp;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitDefinitionMessage;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitDeveloperField;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitDeveloperFieldDescriptionMessage;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitField;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitHeader;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitRecordCompressedTimestampHeader;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitRecordHeader;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.FitRecordNormalHeader;
import org.openstreetmap.josm.plugins.fit.lib.records.internal.IField;
import org.openstreetmap.josm.plugins.fit.lib.utils.CountingInputStream;
import org.openstreetmap.josm.plugins.fit.lib.utils.NumberUtils;

public final class FitReader {
    private static final FitData[] EMPTY_FIT_DATA_ARRAY = new FitData[0];

    private FitReader() {
    }

    public static FitData[] read(InputStream inputStream, FitReaderOptions ... options) throws FitException {
        CountingInputStream bufferedInputStream = new CountingInputStream(inputStream.markSupported() ? inputStream : new BufferedInputStream(inputStream));
        EnumSet<FitReaderOptions> optionsSet = options.length == 0 ? EnumSet.noneOf(FitReaderOptions.class) : EnumSet.of(options[0], options);
        ArrayList<FitData> fitData = new ArrayList<FitData>();
        try {
            FitReader.parseData(bufferedInputStream, optionsSet, fitData);
        }
        catch (FitException fitException) {
            FitReader.handleException(optionsSet, fitException);
        }
        catch (IOException | IllegalArgumentException ioException) {
            FitReader.handleException(optionsSet, new FitException(ioException));
        }
        return fitData.toArray(EMPTY_FIT_DATA_ARRAY);
    }

    private static void parseData(CountingInputStream bufferedInputStream, Set<FitReaderOptions> optionsSet, List<FitData> fitData) throws IOException {
        FitHeader header = FitReader.readFitHeader(bufferedInputStream);
        long headerSize = bufferedInputStream.bytesRead();
        bufferedInputStream.mark(1);
        FitDefinitionMessage[] localMessageHeaders = new FitDefinitionMessage[1];
        FitDeveloperFieldDescriptionMessage[] developerData = new FitDeveloperFieldDescriptionMessage[]{};
        LastTimestamp lastTimeStamp = new LastTimestamp(Long.MIN_VALUE, 0, 0);
        while (bufferedInputStream.read() != -1 && bufferedInputStream.bytesRead() < header.dataSize() + headerSize) {
            bufferedInputStream.reset();
            FitRecordHeader nextRecordHeader = FitReader.readNextRecordHeader(bufferedInputStream);
            if (nextRecordHeader instanceof FitRecordNormalHeader) {
                FitRecordNormalHeader normalHeader = (FitRecordNormalHeader)nextRecordHeader;
                if (normalHeader.isDefinitionMessage()) {
                    localMessageHeaders = FitReader.parseFitRecordDefinitionHeader(bufferedInputStream, localMessageHeaders, normalHeader);
                } else {
                    ParseFitRecordHeader parsed = FitReader.parseFitRecordHeader(optionsSet, bufferedInputStream, localMessageHeaders, developerData, fitData, normalHeader);
                    if (parsed.lastTimestamp() != null) {
                        lastTimeStamp = parsed.lastTimestamp();
                    }
                    developerData = parsed.devFields();
                }
            } else if (nextRecordHeader instanceof FitRecordCompressedTimestampHeader) {
                FitRecordCompressedTimestampHeader compressedHeader = (FitRecordCompressedTimestampHeader)nextRecordHeader;
                if (lastTimeStamp.lastTimestamp() == Long.MIN_VALUE) {
                    FitReader.handleException(optionsSet, new FitException("No full timestamp found before compressed timestamp header"));
                    break;
                }
                if (compressedHeader.timeOffset() < lastTimeStamp.lastOffset()) {
                    lastTimeStamp = new LastTimestamp(lastTimeStamp.lastTimestamp(), lastTimeStamp.offsetAddition() + 1, lastTimeStamp.lastOffset());
                }
                FitReader.parseFitRecordCompressedHeader(optionsSet, bufferedInputStream, localMessageHeaders, developerData, fitData, compressedHeader, lastTimeStamp);
                lastTimeStamp = new LastTimestamp(lastTimeStamp.lastTimestamp(), lastTimeStamp.offsetAddition(), compressedHeader.timeOffset());
            } else {
                FitReader.handleException(optionsSet, new UnsupportedOperationException("Unknown record type: " + String.valueOf(nextRecordHeader.getClass())));
            }
            bufferedInputStream.mark(1);
        }
    }

    private static FitDefinitionMessage[] parseFitRecordDefinitionHeader(InputStream inputStream, FitDefinitionMessage[] localMessageHeaders, FitRecordNormalHeader normalHeader) throws IOException {
        FitDefinitionMessage fitDefinitionMessage = FitReader.readNextDefinition(normalHeader, inputStream);
        if (localMessageHeaders.length <= normalHeader.localMessageType()) {
            localMessageHeaders = Arrays.copyOf(localMessageHeaders, normalHeader.localMessageType() + 1);
        }
        localMessageHeaders[normalHeader.localMessageType()] = fitDefinitionMessage;
        return localMessageHeaders;
    }

    private static ParseFitRecordHeader parseFitRecordHeader(Set<FitReaderOptions> optionsSet, InputStream bufferedInputStream, FitDefinitionMessage[] localMessageHeaders, FitDeveloperFieldDescriptionMessage[] developerData, List<FitData> fitData, FitRecordNormalHeader normalHeader) throws IOException {
        Object obj = FitReader.readNextRecord(localMessageHeaders[normalHeader.localMessageType()], developerData, bufferedInputStream);
        LastTimestamp lastTimeStamp = null;
        if (obj instanceof IFitTimestamp) {
            IFitTimestamp timestamp = (IFitTimestamp)obj;
            lastTimeStamp = new LastTimestamp(timestamp.timestamp().getEpochSecond() & 0xFFFFFFE0L, 0, 0);
        }
        if (obj instanceof FitData) {
            FitData fd = (FitData)obj;
            fitData.add(fd);
        } else if (obj instanceof FitDeveloperFieldDescriptionMessage) {
            FitDeveloperFieldDescriptionMessage developerFieldDescriptionMessage = (FitDeveloperFieldDescriptionMessage)obj;
            if (developerData.length <= developerFieldDescriptionMessage.developerDataIndex()) {
                developerData = Arrays.copyOf(developerData, developerFieldDescriptionMessage.developerDataIndex() + 1);
            }
            developerData[developerFieldDescriptionMessage.developerDataIndex()] = developerFieldDescriptionMessage;
        } else {
            FitReader.handleException(optionsSet, new IllegalStateException("Unknown record type"));
        }
        return new ParseFitRecordHeader(lastTimeStamp, developerData);
    }

    private static void parseFitRecordCompressedHeader(Set<FitReaderOptions> optionsSet, InputStream bufferedInputStream, FitDefinitionMessage[] localMessageHeaders, FitDeveloperFieldDescriptionMessage[] developerData, List<FitData> fitData, FitRecordCompressedTimestampHeader compressedHeader, LastTimestamp lastTimeStamp) throws IOException {
        Object obj = FitReader.readNextRecord(localMessageHeaders[compressedHeader.localMessageType()], developerData, bufferedInputStream);
        if (obj instanceof FitData) {
            FitData fd = (FitData)obj;
            long actualTimestamp = lastTimeStamp.lastTimestamp() + (long)compressedHeader.timeOffset() + (long)lastTimeStamp.offsetAddition() * 32L;
            if (fd instanceof IFitTimestamp) {
                IFitTimestamp timestamp = (IFitTimestamp)((Object)fd);
                fitData.add((FitData)timestamp.withTimestamp(Instant.ofEpochSecond(actualTimestamp)));
            } else {
                fitData.add(fd);
            }
        } else {
            FitReader.handleException(optionsSet, new IllegalStateException("Cannot handle non-data types with compressed timestamp headers"));
        }
    }

    private static <T extends Throwable> void handleException(Set<FitReaderOptions> options, T throwable) throws T {
        if (!options.contains((Object)FitReaderOptions.TRY_TO_FINISH)) {
            throw throwable;
        }
        Logger.getLogger(FitReader.class.getCanonicalName()).log(Level.WARNING, throwable.getMessage(), throwable);
    }

    static FitHeader readFitHeader(InputStream inputStream) throws IOException {
        int size = inputStream.read();
        if (size < 12) {
            throw new FitException("FIT header is not the expected size.");
        }
        int protocolVersion = inputStream.read();
        int profileVersion = inputStream.read() + (inputStream.read() << 8);
        long dataSize = (long)(inputStream.read() + (inputStream.read() << 8) + (inputStream.read() << 16)) + ((long)inputStream.read() << 24);
        byte[] fit = new byte[4];
        int read = inputStream.read(fit);
        if (read != fit.length || fit[0] != 46 || fit[1] != 70 || fit[2] != 73 || fit[3] != 84) {
            throw new FitException("FIT header has the wrong data type: " + new String(fit, StandardCharsets.US_ASCII));
        }
        int crc = size >= 14 ? inputStream.read() + (inputStream.read() << 8) : 0;
        return new FitHeader((short)protocolVersion, profileVersion, dataSize, crc);
    }

    static FitRecordHeader readNextRecordHeader(InputStream inputStream) throws IOException {
        int header = inputStream.read();
        if ((header & 0x80) == 0) {
            return new FitRecordNormalHeader((header & 0x40) != 0, (header & 0x20) != 0, (header & 0x10) != 0, (byte)(header & 0xF));
        }
        return new FitRecordCompressedTimestampHeader((byte)((header & 0x60) >> 6), (byte)(header & 0x1F));
    }

    static FitDefinitionMessage readNextDefinition(FitRecordNormalHeader normalHeader, InputStream inputStream) throws IOException {
        List fitDeveloperFields;
        int reserved = inputStream.read();
        boolean littleEndian = inputStream.read() == 0;
        int globalMessageNumber = NumberUtils.decodeInt(2, littleEndian, inputStream);
        int numberOfFields = inputStream.read();
        ArrayList<FitField> fitFields = new ArrayList<FitField>(Math.max(0, numberOfFields));
        for (int i = 0; i < numberOfFields; ++i) {
            fitFields.add(FitReader.readNextField(inputStream));
        }
        if (normalHeader.messageTypeSpecific()) {
            int numberOfDeveloperFields = inputStream.read();
            fitDeveloperFields = new ArrayList(numberOfDeveloperFields);
            for (int i = 0; i < numberOfDeveloperFields; ++i) {
                fitDeveloperFields.add(FitReader.readNextDeveloperField(inputStream));
            }
        } else {
            fitDeveloperFields = Collections.emptyList();
        }
        return new FitDefinitionMessage((short)reserved, littleEndian, globalMessageNumber, Collections.unmodifiableList(fitFields), Collections.unmodifiableList(fitDeveloperFields));
    }

    static FitField readNextField(InputStream inputStream) throws IOException {
        return FitReader.readNextField(FitField.class, inputStream);
    }

    static FitDeveloperField readNextDeveloperField(InputStream inputStream) throws IOException {
        return FitReader.readNextField(FitDeveloperField.class, inputStream);
    }

    private static <T extends IField> T readNextField(Class<T> clazz, InputStream inputStream) throws IOException {
        int fieldNumber = inputStream.read();
        int size = inputStream.read();
        int baseType = inputStream.read();
        if (fieldNumber == -1 || size == -1 || baseType == -1) {
            throw new FitException("Reached end of FIT file unexpectedly");
        }
        if (clazz == FitDeveloperField.class) {
            return (T)((IField)clazz.cast(new FitDeveloperField((short)fieldNumber, (short)size, (short)baseType)));
        }
        if (clazz == FitField.class) {
            return (T)((IField)clazz.cast(new FitField(fieldNumber, size, baseType)));
        }
        throw new IllegalArgumentException("Unknown type: " + String.valueOf(clazz));
    }

    static Object readNextRecord(FitDefinitionMessage definitionMessage, FitDeveloperFieldDescriptionMessage[] developerFields, InputStream inputStream) throws IOException {
        return Global.parseData(definitionMessage, developerFields, inputStream);
    }

    private record LastTimestamp(long lastTimestamp, int offsetAddition, byte lastOffset) {
    }

    private record ParseFitRecordHeader(LastTimestamp lastTimestamp, FitDeveloperFieldDescriptionMessage[] devFields) {
    }
}

