/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.xml.resolver;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.LinkedList;
import java.util.logging.Logger;
import org.geotools.util.URLs;
import org.geotools.util.logging.Logging;
import org.geotools.xml.resolver.SchemaResolver;

public class SchemaCache {
    private static final Logger LOGGER = Logging.getLogger(SchemaCache.class);
    private static final int DEFAULT_DOWNLOAD_BLOCK_SIZE = 4096;
    private static final boolean DEFAULT_KEEP_QUERY = true;
    private static final String[] GEOSERVER_DATA_DIRECTORY_SUBDIRECTORIES = new String[]{"styles", "workspaces"};
    private static final String CACHE_DIRECTORY_NAME = "app-schema-cache";
    public static final String PROVIDED_CACHE_LOCATION_KEY = "schema.cache.dir";
    private static boolean automaticConfigurationEnabled = true;
    private final File directory;
    private final boolean download;
    private final boolean keepQuery;
    private static int downloadTimeout = 60000;
    private static final int MAX_FOLLOW_REDIRECT;

    public SchemaCache(File directory, boolean download) {
        this(directory, download, false);
    }

    public SchemaCache(File directory, boolean download, boolean keepQuery) {
        this.directory = directory;
        this.download = download;
        this.keepQuery = keepQuery;
    }

    public File getDirectory() {
        return this.directory;
    }

    public File getTempDirectory() {
        try {
            File tempFolder = File.createTempFile("schema", "cache");
            tempFolder.delete();
            tempFolder.mkdirs();
            return tempFolder;
        }
        catch (IOException e) {
            LOGGER.severe("Can't create temporary folder");
            throw new RuntimeException(e);
        }
    }

    public boolean isDownloadAllowed() {
        return this.download;
    }

    static void delete(File file) {
        File[] files;
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            for (File f : files) {
                SchemaCache.delete(f);
            }
        }
        file.delete();
    }

    static void store(File file, byte[] bytes) {
        if (file.getParentFile() != null && !file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
        }
        try (BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(file));){
            ((OutputStream)output).write(bytes);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    static byte[] download(String location) {
        URI locationUri;
        try {
            locationUri = new URI(location);
        }
        catch (URISyntaxException e) {
            return null;
        }
        return SchemaCache.download(locationUri);
    }

    static byte[] download(URI location) {
        return SchemaCache.download(location, 4096);
    }

    static byte[] download(URI location, int blockSize) {
        return SchemaCache.download(location, blockSize, 0);
    }

    static byte[] download(URI location, int blockSize, int redirectionCount) {
        try {
            Object block;
            URL url = location.toURL();
            String protocol = url.getProtocol();
            if (protocol == null || !protocol.equals("http") && !protocol.equals("https")) {
                LOGGER.warning("Unexpected download URL protocol: " + protocol);
                return null;
            }
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setConnectTimeout(downloadTimeout);
            connection.setReadTimeout(downloadTimeout);
            connection.setUseCaches(false);
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                if (SchemaCache.hasRedirect(responseCode)) {
                    return SchemaCache.followRedirect(connection, blockSize, redirectionCount);
                }
                LOGGER.warning(String.format("Unexpected response \"%d %s\" while downloading %s", connection.getResponseCode(), connection.getResponseMessage(), location.toString()));
                return null;
            }
            LinkedList<byte[]> blocks = new LinkedList<byte[]>();
            try (InputStream input = connection.getInputStream();){
                int count;
                while ((count = input.read((byte[])(block = new byte[blockSize]))) != -1) {
                    if (count == blockSize) {
                        blocks.add((byte[])block);
                        continue;
                    }
                    byte[] shortBlock = new byte[count];
                    System.arraycopy(block, 0, shortBlock, 0, count);
                    blocks.add(shortBlock);
                }
            }
            int totalCount = 0;
            block = blocks.iterator();
            while (block.hasNext()) {
                byte[] b = (byte[])block.next();
                totalCount += b.length;
            }
            byte[] bytes = new byte[totalCount];
            int position = 0;
            for (byte[] b : blocks) {
                System.arraycopy(b, 0, bytes, position, b.length);
                position += b.length;
            }
            return bytes;
        }
        catch (Exception e) {
            throw new RuntimeException("Error downloading location: " + location.toString(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String resolveLocation(String location) {
        File file;
        String path = SchemaResolver.getSimpleHttpResourcePath(location, this.keepQuery);
        if (path == null) {
            return null;
        }
        String relativePath = path.substring(1);
        try {
            file = new File(this.getDirectory(), relativePath).getCanonicalFile();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        Class<SchemaCache> e = SchemaCache.class;
        synchronized (SchemaCache.class) {
            if (file.exists()) {
                // ** MonitorExit[e] (shouldn't be in output)
                return URLs.fileToUrl(file).toExternalForm();
            }
            // ** MonitorExit[e] (shouldn't be in output)
            if (this.isDownloadAllowed()) {
                byte[] bytes = SchemaCache.download(location);
                if (bytes == null) {
                    return null;
                }
                Class<SchemaCache> clazz = SchemaCache.class;
                synchronized (SchemaCache.class) {
                    if (!file.exists()) {
                        SchemaCache.store(file, bytes);
                        LOGGER.info("Cached XML schema: " + location);
                    }
                    // ** MonitorExit[var6_8] (shouldn't be in output)
                    return URLs.fileToUrl(file).toExternalForm();
                }
            }
            return null;
        }
    }

    public static SchemaCache buildAutomaticallyConfiguredUsingFileUrl(URL url) {
        if (!automaticConfigurationEnabled) {
            return null;
        }
        SchemaCache provided = SchemaCache.getProvidedSchemaCache();
        if (provided != null) {
            return provided;
        }
        File file = URLs.urlToFile(url);
        while (true) {
            if (file == null) {
                LOGGER.warning("Automatic app-schema-cache directory build failed, Geoserver root folder or app-schema-cache folder not found");
                return null;
            }
            if (SchemaCache.isSuitableDirectoryToContainCache(file)) {
                return new SchemaCache(new File(file, CACHE_DIRECTORY_NAME), true, true);
            }
            file = file.getParentFile();
        }
    }

    private static SchemaCache getProvidedSchemaCache() {
        String directory = System.getProperty(PROVIDED_CACHE_LOCATION_KEY);
        if (directory == null) {
            return null;
        }
        File cacheDirectory = new File(directory);
        if (cacheDirectory.exists() && cacheDirectory.isFile()) {
            throw new RuntimeException(String.format("Provided schema cache directory '%s' already exists but it's a file.", cacheDirectory.getAbsolutePath()));
        }
        if (!cacheDirectory.exists()) {
            cacheDirectory.mkdir();
        }
        LOGGER.fine(String.format("Using provided schema cache directory '%s'.", cacheDirectory.getAbsolutePath()));
        return new SchemaCache(cacheDirectory, true, true);
    }

    public static void disableAutomaticConfiguration() {
        automaticConfigurationEnabled = false;
    }

    public static void enableAutomaticConfiguration() {
        automaticConfigurationEnabled = true;
    }

    public static boolean isAutomaticConfigurationEnabled() {
        return automaticConfigurationEnabled;
    }

    static boolean isSuitableDirectoryToContainCache(File directory) {
        if (!directory.isDirectory()) {
            return false;
        }
        if (new File(directory, CACHE_DIRECTORY_NAME).isDirectory()) {
            return true;
        }
        for (String subdirectory : GEOSERVER_DATA_DIRECTORY_SUBDIRECTORIES) {
            File dir = new File(directory, subdirectory);
            if (dir.isDirectory()) continue;
            return false;
        }
        File workspacesDir = new File(directory, "workspaces");
        File defaultXmlFile = new File(workspacesDir, "default.xml");
        return defaultXmlFile.exists();
    }

    private static boolean hasRedirect(int responseCode) {
        return responseCode == 303 || responseCode == 301 || responseCode == 302;
    }

    private static byte[] followRedirect(HttpURLConnection connection, int blockSize, int redirectionCount) {
        String redirect = connection.getHeaderField("Location");
        byte[] result = null;
        if (redirect == null) {
            LOGGER.warning("Tried to follow redirect but no url was provided in Location header");
        } else if (redirectionCount <= MAX_FOLLOW_REDIRECT) {
            ++redirectionCount;
            try {
                URI redirectURL = new URI(redirect);
                LOGGER.info("Following redirect to " + redirect);
                result = SchemaCache.download(redirectURL, blockSize, redirectionCount);
            }
            catch (URISyntaxException uri) {
                LOGGER.warning("Tried to follow redirect but invalid url was provided in Location header: " + redirect);
            }
        } else {
            LOGGER.warning("Max number of follow redirect attempts (" + MAX_FOLLOW_REDIRECT + ") reached. Returning null");
        }
        return result;
    }

    static {
        if (System.getProperty("schema.cache.download.timeout") != null) {
            try {
                downloadTimeout = Integer.parseInt(System.getProperty("schema.cache.download.timeout"));
            }
            catch (NumberFormatException e) {
                LOGGER.warning("schema.cache.download.timeout has a wrong format: should be a number");
            }
        }
        MAX_FOLLOW_REDIRECT = Integer.getInteger("org.geotools.xml.schema.maxFollowRedirect", 16);
    }
}

