/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.shapefile.shp;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import org.geotools.data.shapefile.files.StreamLogging;
import org.geotools.data.shapefile.shp.ShapeHandler;
import org.geotools.data.shapefile.shp.ShapeType;
import org.geotools.data.shapefile.shp.ShapefileException;
import org.geotools.data.shapefile.shp.ShapefileHeader;
import org.geotools.util.NIOUtilities;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;

public class ShapefileWriter {
    FileChannel shpChannel;
    FileChannel shxChannel;
    ByteBuffer shapeBuffer;
    ByteBuffer indexBuffer;
    ShapeHandler handler;
    ShapeType type;
    int offset;
    int lp;
    int cnt;
    private StreamLogging shpLogger = new StreamLogging("SHP Channel in ShapefileWriter");
    private StreamLogging shxLogger = new StreamLogging("SHX Channel in ShapefileWriter");
    private GeometryFactory gf = new GeometryFactory();

    public ShapefileWriter(FileChannel shpChannel, FileChannel shxChannel) throws IOException {
        this.shpChannel = shpChannel;
        this.shxChannel = shxChannel;
        this.shpLogger.open();
        this.shxLogger.open();
    }

    private void allocateBuffers() {
        this.shapeBuffer = NIOUtilities.allocate(16384);
        this.indexBuffer = NIOUtilities.allocate(100);
    }

    private void checkShapeBuffer(int size) {
        if (this.shapeBuffer != null && this.shapeBuffer.capacity() < size) {
            NIOUtilities.clean(this.shapeBuffer, false);
            this.shapeBuffer = NIOUtilities.allocate(size);
        }
    }

    private void drain() throws IOException {
        this.shapeBuffer.flip();
        this.indexBuffer.flip();
        while (this.shapeBuffer.remaining() > 0) {
            this.shpChannel.write(this.shapeBuffer);
        }
        while (this.indexBuffer.remaining() > 0) {
            this.shxChannel.write(this.indexBuffer);
        }
        this.shapeBuffer.flip().limit(this.shapeBuffer.capacity());
        this.indexBuffer.flip().limit(this.indexBuffer.capacity());
    }

    private void writeHeaders(GeometryCollection geometries, ShapeType type) throws IOException {
        int fileLength = 100;
        for (int i = geometries.getNumGeometries() - 1; i >= 0; --i) {
            int size = this.handler.getLength(geometries.getGeometryN(i)) + 8;
            fileLength += size;
        }
        this.writeHeaders(geometries.getEnvelopeInternal(), type, geometries.getNumGeometries(), fileLength);
    }

    public void writeHeaders(Envelope bounds, ShapeType type, int numberOfGeometries, int fileLength) throws IOException {
        try {
            this.handler = type.getShapeHandler(this.gf);
        }
        catch (ShapefileException se) {
            throw new RuntimeException("unexpected Exception", se);
        }
        if (this.shapeBuffer == null) {
            this.allocateBuffers();
        }
        ShapefileHeader header = new ShapefileHeader();
        header.write(this.shapeBuffer, type, numberOfGeometries, fileLength / 2, bounds.getMinX(), bounds.getMinY(), bounds.getMaxX(), bounds.getMaxY());
        header.write(this.indexBuffer, type, numberOfGeometries, 50 + 4 * numberOfGeometries, bounds.getMinX(), bounds.getMinY(), bounds.getMaxX(), bounds.getMaxY());
        this.offset = 50;
        this.type = type;
        this.cnt = 0;
        this.shpChannel.position(0L);
        this.shxChannel.position(0L);
        this.drain();
    }

    public void skipHeaders() throws IOException {
        if (this.shapeBuffer == null) {
            this.allocateBuffers();
        }
        this.shpChannel.position(100L);
        this.shxChannel.position(100L);
    }

    public void writeGeometry(Geometry g) throws IOException {
        if (this.shapeBuffer == null) {
            throw new IOException("Must write headers first");
        }
        this.lp = this.shapeBuffer.position();
        int length = g == null ? this.writeNullGeometry() : this.writeNonNullGeometry(g);
        assert (length * 2 == this.shapeBuffer.position() - this.lp - 8);
        this.lp = this.shapeBuffer.position();
        this.indexBuffer.putInt(this.offset);
        this.indexBuffer.putInt(length);
        this.offset += length + 4;
        this.drain();
        assert (this.shapeBuffer.position() == 0);
    }

    private int writeNonNullGeometry(Geometry g) {
        int length = this.handler.getLength(g);
        this.checkShapeBuffer(length + 8);
        this.shapeBuffer.order(ByteOrder.BIG_ENDIAN);
        this.shapeBuffer.putInt(++this.cnt);
        this.shapeBuffer.putInt(length /= 2);
        this.shapeBuffer.order(ByteOrder.LITTLE_ENDIAN);
        this.shapeBuffer.putInt(this.type.id);
        this.handler.write(this.shapeBuffer, g);
        return length;
    }

    protected int writeNullGeometry() throws IOException {
        int length = 4;
        this.checkShapeBuffer(8 + length);
        this.shapeBuffer.order(ByteOrder.BIG_ENDIAN);
        this.shapeBuffer.putInt(++this.cnt);
        this.shapeBuffer.putInt(length /= 2);
        this.shapeBuffer.order(ByteOrder.LITTLE_ENDIAN);
        this.shapeBuffer.putInt(ShapeType.NULL.id);
        return length;
    }

    public void close() throws IOException {
        try {
            if (this.shpChannel != null && this.shpChannel.isOpen()) {
                this.shpChannel.close();
                this.shpLogger.close();
            }
        }
        finally {
            if (this.shxChannel != null && this.shxChannel.isOpen()) {
                this.shxChannel.close();
                this.shxLogger.close();
            }
        }
        this.shpChannel = null;
        this.shxChannel = null;
        this.handler = null;
        if (this.indexBuffer != null) {
            NIOUtilities.clean(this.indexBuffer, false);
        }
        if (this.shapeBuffer != null) {
            NIOUtilities.clean(this.shapeBuffer, false);
        }
        this.indexBuffer = null;
        this.shapeBuffer = null;
    }

    public void write(GeometryCollection geometries, ShapeType type) throws IOException, ShapefileException {
        this.handler = type.getShapeHandler(this.gf);
        this.writeHeaders(geometries, type);
        this.lp = this.shapeBuffer.position();
        int ii = geometries.getNumGeometries();
        for (int i = 0; i < ii; ++i) {
            Geometry g = geometries.getGeometryN(i);
            this.writeGeometry(g);
        }
        this.close();
    }
}

