/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.shapefile.dbf;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.Charset;
import java.text.FieldPosition;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;
import org.geotools.data.shapefile.StreamLogging;
import org.geotools.data.shapefile.dbf.DbaseFileException;
import org.geotools.data.shapefile.dbf.DbaseFileHeader;
import org.geotools.resources.NIOUtilities;

public class DbaseFileWriter {
    private DbaseFileHeader header;
    private FieldFormatter formatter;
    WritableByteChannel channel;
    private ByteBuffer buffer;
    private final byte[][] nullValues;
    private StreamLogging streamLogger = new StreamLogging("Dbase File Writer");
    private Charset charset;
    private TimeZone timeZone;

    public DbaseFileWriter(DbaseFileHeader header, WritableByteChannel out) throws IOException {
        this(header, out, null, null);
    }

    public DbaseFileWriter(DbaseFileHeader header, WritableByteChannel out, Charset charset) throws IOException {
        this(header, out, charset, null);
    }

    public DbaseFileWriter(DbaseFileHeader header, WritableByteChannel out, Charset charset, TimeZone timeZone) throws IOException {
        header.writeHeader(out);
        this.header = header;
        this.channel = out;
        this.charset = charset == null ? Charset.defaultCharset() : charset;
        this.timeZone = timeZone == null ? TimeZone.getDefault() : timeZone;
        this.formatter = new FieldFormatter(this.charset, this.timeZone);
        this.streamLogger.open();
        this.nullValues = new byte[header.getNumFields()][];
        for (int i = 0; i < this.nullValues.length; ++i) {
            int nullChar;
            switch (header.getFieldType(i)) {
                case 'C': 
                case 'G': 
                case 'M': 
                case 'c': {
                    nullChar = 0;
                    break;
                }
                case 'L': 
                case 'l': {
                    nullChar = 63;
                    break;
                }
                case 'F': 
                case 'N': 
                case 'f': 
                case 'n': {
                    nullChar = 42;
                    break;
                }
                case 'D': 
                case 'd': {
                    nullChar = 48;
                    break;
                }
                case '@': {
                    nullChar = 0;
                    break;
                }
                default: {
                    nullChar = 48;
                }
            }
            this.nullValues[i] = new byte[header.getFieldLength(i)];
            Arrays.fill(this.nullValues[i], (byte)nullChar);
        }
        this.buffer = NIOUtilities.allocate(header.getRecordLength());
    }

    private void write() throws IOException {
        this.buffer.position(0);
        int r = this.buffer.remaining();
        while ((r -= this.channel.write(this.buffer)) > 0) {
        }
    }

    public void write(Object[] record) throws IOException, DbaseFileException {
        if (record.length != this.header.getNumFields()) {
            throw new DbaseFileException("Wrong number of fields " + record.length + " expected " + this.header.getNumFields());
        }
        this.buffer.position(0);
        this.buffer.put((byte)32);
        for (int i = 0; i < this.header.getNumFields(); ++i) {
            byte[] bytes;
            if (record[i] == null) {
                bytes = this.nullValues[i];
            } else {
                bytes = this.fieldBytes(record[i], i);
                if (bytes.length != this.nullValues[i].length) {
                    bytes = this.nullValues[i];
                }
            }
            this.buffer.put(bytes);
        }
        this.write();
    }

    private byte[] fieldBytes(Object obj, int col) throws UnsupportedEncodingException {
        String o;
        int fieldLen = this.header.getFieldLength(col);
        switch (this.header.getFieldType(col)) {
            case 'C': 
            case 'c': {
                o = this.formatter.getFieldString(fieldLen, obj.toString());
                break;
            }
            case 'L': 
            case 'l': {
                if (obj instanceof Boolean) {
                    o = (Boolean)obj != false ? "T" : "F";
                    break;
                }
                o = "?";
                break;
            }
            case 'G': 
            case 'M': {
                o = this.formatter.getFieldString(fieldLen, obj.toString());
                break;
            }
            case 'N': 
            case 'n': {
                if (this.header.getFieldDecimalCount(col) == 0) {
                    o = this.formatter.getFieldString(fieldLen, 0, (Number)obj);
                    break;
                }
            }
            case 'F': 
            case 'f': {
                o = this.formatter.getFieldString(fieldLen, this.header.getFieldDecimalCount(col), (Number)obj);
                break;
            }
            case 'D': 
            case 'd': {
                if (obj instanceof Calendar) {
                    o = this.formatter.getFieldString(((Calendar)obj).getTime());
                    break;
                }
                o = this.formatter.getFieldString((Date)obj);
                break;
            }
            case '@': {
                o = this.formatter.getFieldStringDateTime((Date)obj);
                if (!Boolean.getBoolean("org.geotools.shapefile.datetime")) break;
                char[] carr = o.toCharArray();
                byte[] barr = new byte[carr.length];
                for (int i = 0; i < carr.length; ++i) {
                    barr[i] = (byte)carr[i];
                }
                return barr;
            }
            default: {
                throw new RuntimeException("Unknown type " + this.header.getFieldType(col));
            }
        }
        return o.getBytes(this.charset.name());
    }

    public void close() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.close();
            this.streamLogger.close();
        }
        if (this.buffer != null) {
            NIOUtilities.clean(this.buffer, false);
        }
        this.buffer = null;
        this.channel = null;
        this.formatter = null;
    }

    public static class FieldFormatter {
        private StringBuffer buffer = new StringBuffer(255);
        private NumberFormat numFormat = NumberFormat.getNumberInstance(Locale.US);
        private Calendar calendar;
        private final long MILLISECS_PER_DAY = 86400000L;
        private String emptyString;
        private static final int MAXCHARS = 255;
        private Charset charset;

        public FieldFormatter(Charset charset, TimeZone timeZone) {
            this.numFormat.setGroupingUsed(false);
            StringBuffer sb = new StringBuffer(255);
            sb.setLength(255);
            for (int i = 0; i < 255; ++i) {
                sb.setCharAt(i, ' ');
            }
            this.charset = charset;
            this.calendar = Calendar.getInstance(timeZone, Locale.US);
            this.emptyString = sb.toString();
        }

        public String getFieldString(int size, String s) {
            try {
                this.buffer.replace(0, size, this.emptyString);
                this.buffer.setLength(size);
                int maxSize = size;
                if (s != null) {
                    this.buffer.replace(0, size, s);
                    int currentBytes = s.substring(0, Math.min(size, s.length())).getBytes(this.charset.name()).length;
                    if (currentBytes > size) {
                        char[] c = new char[1];
                        int index = size - 1;
                        while (currentBytes > size) {
                            c[0] = this.buffer.charAt(index);
                            String string = new String(c);
                            this.buffer.deleteCharAt(index);
                            currentBytes -= string.getBytes().length;
                            --maxSize;
                            --index;
                        }
                    } else if (s.length() < size) {
                        maxSize = size - (currentBytes - s.length());
                        for (int i = s.length(); i < size; ++i) {
                            this.buffer.append(' ');
                        }
                    }
                }
                this.buffer.setLength(maxSize);
                return this.buffer.toString();
            }
            catch (UnsupportedEncodingException e) {
                throw new RuntimeException("This error should never occurr", e);
            }
        }

        public String getFieldString(Date d) {
            if (d != null) {
                this.buffer.delete(0, this.buffer.length());
                this.calendar.setTime(d);
                int year = this.calendar.get(1);
                int month = this.calendar.get(2) + 1;
                int day = this.calendar.get(5);
                if (year < 1000) {
                    if (year >= 100) {
                        this.buffer.append("0");
                    } else if (year >= 10) {
                        this.buffer.append("00");
                    } else {
                        this.buffer.append("000");
                    }
                }
                this.buffer.append(year);
                if (month < 10) {
                    this.buffer.append("0");
                }
                this.buffer.append(month);
                if (day < 10) {
                    this.buffer.append("0");
                }
                this.buffer.append(day);
            } else {
                this.buffer.setLength(8);
                this.buffer.replace(0, 8, this.emptyString);
            }
            this.buffer.setLength(8);
            return this.buffer.toString();
        }

        public String getFieldStringDateTime(Date d) {
            if (d == null) {
                return null;
            }
            long difference = d.getTime() - DbaseFileHeader.MILLIS_SINCE_4713;
            int days = (int)(difference / 86400000L);
            int time = (int)(difference % 86400000L);
            try {
                ByteArrayOutputStream o_bytes = new ByteArrayOutputStream();
                DataOutputStream o_stream = new DataOutputStream(new BufferedOutputStream(o_bytes));
                o_stream.writeInt(days);
                o_stream.writeInt(time);
                o_stream.flush();
                byte[] bytes = o_bytes.toByteArray();
                char[] out = new char[]{(char)bytes[3], (char)bytes[2], (char)bytes[1], (char)bytes[0], (char)bytes[7], (char)bytes[6], (char)bytes[5], (char)bytes[4]};
                return new String(out);
            }
            catch (IOException e) {
                return null;
            }
        }

        public String getFieldString(int size, int decimalPlaces, Number n) {
            int diff;
            this.buffer.delete(0, this.buffer.length());
            if (n != null) {
                this.numFormat.setMaximumFractionDigits(decimalPlaces);
                this.numFormat.setMinimumFractionDigits(decimalPlaces);
                this.numFormat.format(n, this.buffer, new FieldPosition(0));
            }
            if ((diff = size - this.buffer.length()) >= 0) {
                while (diff-- > 0) {
                    this.buffer.insert(0, ' ');
                }
            } else {
                this.buffer.setLength(size);
            }
            return this.buffer.toString();
        }
    }
}

