// License: GPL. For details, see LICENSE file.
package org.openstreetmap.josm.io;

import static org.openstreetmap.josm.tools.I18n.tr;

import java.io.File;
import java.io.IOException;
import java.util.List;

import javax.swing.JOptionPane;

import org.openstreetmap.josm.Main;
import org.openstreetmap.josm.actions.ExtensionFileFilter;

public abstract class FileImporter implements Comparable<FileImporter> {

    public ExtensionFileFilter filter;

    public FileImporter(ExtensionFileFilter filter) {
        this.filter = filter;
    }

    public boolean acceptFile(File pathname) {
        return filter.acceptName(pathname.getName());
    }

    /**
     * A batch importer is a file importer that prefers to read multiple files at the same time.
     */
    public boolean isBatchImporter() {
        return false;
    }
    
    /**
     * Needs to be implemented if isBatchImporter() returns false.
     */
    public void importData(File file) throws IOException, IllegalDataException {
        throw new IOException(tr("Could not import ''{0}''.", file.getName()));
    }

    /**
     * Needs to be implemented if isBatchImporter() returns true.
     */
    public void importData(List<File> files) throws IOException, IllegalDataException {
        throw new IOException(tr("Could not import Files."));
    }

    /**
     * Wrapper to give meaningful output if things go wrong.
     */
    public void importDataHandleExceptions(File f) {
        try {
            System.out.println("Open file: " + f.getAbsolutePath() + " (" + f.length() + " bytes)");
            importData(f);
        } catch (Exception e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(
                    Main.parent,
                    tr("<html>Could not read file ''{0}\''.<br> Error is: <br>{1}</html>", f.getName(), e.getMessage()),
                    tr("Error"),
                    JOptionPane.ERROR_MESSAGE
            );
        }
    }
    public void importDataHandleExceptions(List<File> files) {
        try {
            System.out.println("Open "+files.size()+" files");
            importData(files);
        } catch (Exception e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(
                    Main.parent,
                    tr("<html>Could not read files.<br> Error is: <br>{0}</html>", e.getMessage()),
                    tr("Error"),
                    JOptionPane.ERROR_MESSAGE
            );
        }
    }

    /**
     * If multiple files (with multiple file formats) are selected, 
     * they are opened in the order of their priorities.
     * Highest priority comes first.
     */
    public double getPriority() {
        return 0;
    }

    public int compareTo(FileImporter other) {
        return (new Double(this.getPriority())).compareTo(other.getPriority());
    }
}
