package relcontext;

import java.awt.event.ActionEvent;
import java.awt.event.ComponentEvent;
import java.awt.event.FocusEvent;
import java.beans.PropertyChangeEvent;
import static org.openstreetmap.josm.tools.I18n.tr;

import java.awt.BorderLayout;
import java.awt.Dialog.ModalityType;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.FocusAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseAdapter;
import java.beans.PropertyChangeListener;

import org.openstreetmap.josm.Main;
import org.openstreetmap.josm.data.SelectionChangedListener;
import org.openstreetmap.josm.data.osm.Relation;
import org.openstreetmap.josm.data.osm.event.DatasetEventManager.FireMode;
import org.openstreetmap.josm.data.osm.event.SelectionEventManager;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.gui.layer.OsmDataLayer;
import org.openstreetmap.josm.gui.MapView;
import org.openstreetmap.josm.gui.MapView.EditLayerChangeListener;
import org.openstreetmap.josm.gui.OsmPrimitivRenderer;
import org.openstreetmap.josm.gui.dialogs.ToggleDialog;
import org.openstreetmap.josm.tools.Shortcut;

import java.util.*;
import javax.swing.*;
import org.openstreetmap.josm.command.ChangeCommand;
import org.openstreetmap.josm.data.osm.RelationMember;
import org.openstreetmap.josm.data.osm.event.DatasetEventManager;
import org.openstreetmap.josm.gui.tagging.ac.AutoCompletingComboBox;
import org.openstreetmap.josm.gui.tagging.ac.AutoCompletionListItem;
import org.openstreetmap.josm.gui.tagging.ac.AutoCompletionManager;
import org.openstreetmap.josm.tools.GBC;
import relcontext.actions.*;

/**
 * The new, advanced relation editing panel.
 * 
 * @author Zverik
 */
public class RelContextDialog extends ToggleDialog implements EditLayerChangeListener, ChosenRelationListener, SelectionChangedListener {
    private JList relationsList;
    private final DefaultListModel relationsData;
    private ChosenRelation chosenRelation;
    private JPanel chosenRelationPanel;
    private ChosenRelationPopupMenu popupMenu;
    private JLabel crRoleIndicator;
    private AutoCompletingComboBox roleBox;
    private String lastSelectedRole;

    public RelContextDialog() {
        super(tr("Advanced Relation Editor"), "icon_relcontext",
                tr("Opens advanced relation/multipolygon editor panel"),
                Shortcut.registerShortcut("view:relcontext", tr("Toggle: {0}", tr("Open Relation Editor")),
                KeyEvent.VK_R, Shortcut.GROUP_LAYER, Shortcut.SHIFT_DEFAULT), 150);

        JPanel rcPanel = new JPanel(new BorderLayout());

        chosenRelation = new ChosenRelation();
        chosenRelation.addChosenRelationListener(this);
        MapView.addEditLayerChangeListener(chosenRelation);

        relationsData = new DefaultListModel();
        relationsList = new JList(relationsData);
        relationsList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        relationsList.setCellRenderer(new OsmPrimitivRenderer() {
            @Override
            protected String getComponentToolTipText( OsmPrimitive value ) {
                return null;
            }
        });
        relationsList.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked( MouseEvent e ) {
                if( Main.main.getEditLayer() == null ) {
                    return;
                }
                chosenRelation.set((Relation)relationsList.getSelectedValue());
                relationsList.clearSelection();
            }
        });
        rcPanel.add(new JScrollPane(relationsList), BorderLayout.CENTER);

        chosenRelationPanel = new JPanel(new BorderLayout());

        // [^] roles [new role][V][Apply]
        final JPanel rolePanel = new JPanel(new GridBagLayout());
        final JButton toggleRolePanelButtonTop = new JButton(new TogglePanelAction(rolePanel));
        final JButton toggleRolePanelButtonIn = new JButton(new TogglePanelAction(rolePanel));
        rolePanel.add(toggleRolePanelButtonIn, GBC.std());
        crRoleIndicator = new JLabel();
        rolePanel.add(crRoleIndicator, GBC.std().insets(5, 0, 5, 0));
        roleBox = new AutoCompletingComboBox();
        roleBox.setEditable(false);
        rolePanel.add(roleBox, GBC.std().fill(GBC.HORIZONTAL));
        rolePanel.add(new JButton(new ApplyNewRoleAction()), GBC.std());
        rolePanel.add(new JButton(new EnterNewRoleAction()), GBC.eol());
//        rolePanel.setVisible(false); // todo: take from preferences

        // [±][X] relation U [AZ][Down][Edit]
        JPanel topLine = new JPanel(new BorderLayout());
        JPanel topLeftButtons = new JPanel(new FlowLayout(FlowLayout.LEFT));
        topLeftButtons.add(new JButton(new AddRemoveMemberAction(chosenRelation)));
        topLeftButtons.add(toggleRolePanelButtonTop);
        topLeftButtons.add(new JButton(new ClearChosenRelationAction(chosenRelation)));
        topLine.add(topLeftButtons, BorderLayout.WEST);
        final ChosenRelationComponent chosenRelationComponent = new ChosenRelationComponent(chosenRelation);
        chosenRelationComponent.addMouseListener(new ChosenRelationMouseAdapter());
        topLine.add(chosenRelationComponent, BorderLayout.CENTER);
        JPanel topRightButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));
        final Action sortAndFixAction = new SortAndFixAction(chosenRelation);
        final JButton sortAndFixButton = new JButton(sortAndFixAction);
        topRightButtons.add(sortAndFixButton);
        final Action downloadChosenRelationAction = new DownloadChosenRelationAction(chosenRelation);
        final JButton downloadButton = new JButton(downloadChosenRelationAction);
        topRightButtons.add(downloadButton);
        topRightButtons.add(new JButton(new EditChosenRelationAction(chosenRelation)));
        topLine.add(topRightButtons, BorderLayout.EAST);

        chosenRelationPanel.add(topLine, BorderLayout.CENTER);
        chosenRelationPanel.add(rolePanel, BorderLayout.SOUTH);
        rcPanel.add(chosenRelationPanel, BorderLayout.NORTH);

        rolePanel.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentHidden( ComponentEvent e ) {
                toggleRolePanelButtonTop.setVisible(true);
            }

            @Override
            public void componentShown( ComponentEvent e ) {
                toggleRolePanelButtonTop.setVisible(false);
            }
        });
        toggleRolePanelButtonTop.setVisible(!rolePanel.isVisible());

        sortAndFixAction.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange( PropertyChangeEvent evt ) {
                sortAndFixButton.setVisible(sortAndFixAction.isEnabled());
            }
        });
        sortAndFixButton.setVisible(false);

        downloadChosenRelationAction.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange( PropertyChangeEvent evt ) {
                downloadButton.setVisible(downloadChosenRelationAction.isEnabled());
            }
        });
        downloadButton.setVisible(false);
        chosenRelationPanel.setVisible(false);

        // [+][Multi] [X]Adm [X]Tags [X]1
        JPanel bottomLine = new JPanel(new GridBagLayout());
        bottomLine.add(new JButton(new CreateRelationAction(chosenRelation)), GBC.std());
        bottomLine.add(new JButton(new CreateMultipolygonAction(chosenRelation)), GBC.std());
        bottomLine.add(Box.createHorizontalGlue(), GBC.std().fill());
        bottomLine.add(new JButton(new FindRelationAction(chosenRelation)), GBC.eol());
        rcPanel.add(bottomLine, BorderLayout.SOUTH);

        add(rcPanel, BorderLayout.CENTER);

        popupMenu = new ChosenRelationPopupMenu();
    }

    @Override
    public void hideNotify() {
        SelectionEventManager.getInstance().removeSelectionListener(this);
        MapView.removeEditLayerChangeListener(this);
        DatasetEventManager.getInstance().removeDatasetListener(chosenRelation);
    }

    @Override
    public void showNotify() {
        SelectionEventManager.getInstance().addSelectionListener(this, FireMode.IN_EDT_CONSOLIDATED);
        MapView.addEditLayerChangeListener(this);
        DatasetEventManager.getInstance().addDatasetListener(chosenRelation, FireMode.IN_EDT);
    }

    public ChosenRelation getChosenRelation() {
        return chosenRelation;
    }

    public void chosenRelationChanged( Relation oldRelation, Relation newRelation ) {
        if( chosenRelationPanel != null )
            chosenRelationPanel.setVisible(newRelation != null);
        if( oldRelation != newRelation && Main.main.getCurrentDataSet() != null )
            selectionChanged(Main.main.getCurrentDataSet().getSelected());
        updateRoleIndicator();
        updateRoleAutoCompletionList();
        // ?
    }

    private void updateRoleIndicator() {
        if( crRoleIndicator == null )
            return;
        String role = "";
        if( chosenRelation != null && chosenRelation.get() != null && Main.main.getCurrentDataSet() != null && !Main.main.getCurrentDataSet().selectionEmpty() ) {
            Collection<OsmPrimitive> selected = Main.main.getCurrentDataSet().getSelected();
            for( RelationMember m : chosenRelation.get().getMembers() ) {
                if( selected.contains(m.getMember()) ) {
                    if( role.length() == 0 && m.getRole() != null )
                        role = m.getRole();
                    else if( !role.equals(m.getRole()) ) {
                        role = tr("<different>");
                        break;
                    }
                }
            }
        }
        crRoleIndicator.setText(role);
    }

    public void selectionChanged( Collection<? extends OsmPrimitive> newSelection ) {
        if( !isVisible() || relationsData == null )
            return;
        updateRoleIndicator();
        // repopulate relations table
        relationsData.clear();
        if( newSelection == null )
            return;
        Set<Relation> rels = new HashSet<Relation>();
        for( OsmPrimitive element : newSelection ) {
            for( OsmPrimitive ref : element.getReferrers() ) {
                if( ref instanceof Relation && !ref.isIncomplete() && !ref.isDeleted() ) {
                    rels.add((Relation) ref);
                }
            }
        }
        for( Relation rel : rels )
            relationsData.addElement(rel);
    }

    private void updateSelection() {
        if (Main.main.getCurrentDataSet() == null) {
            selectionChanged(Collections.<OsmPrimitive>emptyList());
        } else {
            selectionChanged(Main.main.getCurrentDataSet().getSelected());
        }
    }

    public void editLayerChanged( OsmDataLayer oldLayer, OsmDataLayer newLayer ) {
        updateSelection();
    }

    private static final Map<String, String[]> possibleRoles = new HashMap<String, String[]>();
    {
        possibleRoles.put("boundary", new String[] {"admin_centre", "label", "subarea"});
        possibleRoles.put("route", new String[] {"forward", "backward", "stop", "platform"});
    }

    private void updateRoleAutoCompletionList() {
        String currentRole = roleBox.getSelectedItem() == null ? null : ((AutoCompletionListItem)roleBox.getSelectedItem()).getValue();
        List<String> items = new ArrayList<String>();
        items.add("");
        if( chosenRelation != null && chosenRelation.get() != null ) {
            if( chosenRelation.isMultipolygon() ) {
                items.add("outer");
                items.add("inner");
            }
            if( chosenRelation.get().get("type") != null ) {
                String[] values = possibleRoles.get(chosenRelation.get().get("type"));
                if( values != null )
                    for( String value : values )
                        items.add(value);
            }
            for( RelationMember m : chosenRelation.get().getMembers() )
                if( !items.contains(m.getRole()) )
                    items.add(m.getRole());
            lastSelectedRole = currentRole;
        } else if( currentRole != null && currentRole.length() > 0 ) {
            lastSelectedRole = currentRole;
        }
        roleBox.setPossibleItems(items);
        // todo: store last non-empty role and try to select it if present, but do not store if empty role was chosen
//        if( lastSelectedRole != null && items.contains(lastSelectedRole) )
//            roleBox.setSelectedItem(lastSelectedRole);
//        else if( lastSelectedRole == null && items.contains("outer") )
//            roleBox.setSelectedItem("outer");
    }

    private String askForRoleName() {
        JPanel panel = new JPanel(new GridBagLayout());

        final AutoCompletingComboBox role = new AutoCompletingComboBox();
        List<AutoCompletionListItem> items = new ArrayList<AutoCompletionListItem>();
        for( int i = 0; i < roleBox.getModel().getSize(); i++ )
            items.add((AutoCompletionListItem)roleBox.getModel().getElementAt(i));
        role.setPossibleACItems(items);
        role.setEditable(true);

        panel.add(new JLabel(tr("Role")), GBC.std());
        panel.add(Box.createHorizontalStrut(10), GBC.std());
        panel.add(role, GBC.eol().fill(GBC.HORIZONTAL));

        final JOptionPane optionPane = new JOptionPane(panel, JOptionPane.QUESTION_MESSAGE, JOptionPane.OK_CANCEL_OPTION) {
            @Override
            public void selectInitialValue() {
                role.requestFocusInWindow();
                role.getEditor().selectAll();
            }
        };
        final JDialog dlg = optionPane.createDialog(Main.parent, tr("Specify role"));
        dlg.setModalityType(ModalityType.DOCUMENT_MODAL);

        role.getEditor().addActionListener(new ActionListener() {
            public void actionPerformed( ActionEvent e ) {
                dlg.setVisible(false);
                optionPane.setValue(JOptionPane.OK_OPTION);
            }
        });

        dlg.setVisible(true);

        Object answer = optionPane.getValue();
        if( answer == null || answer == JOptionPane.UNINITIALIZED_VALUE
                || (answer instanceof Integer && (Integer)answer != JOptionPane.OK_OPTION) ) {
            return null;
        }

        return role.getEditor().getItem().toString().trim();
    }

    private class ChosenRelationMouseAdapter extends MouseAdapter {
        @Override
        public void mouseClicked( MouseEvent e ) {
            if( SwingUtilities.isLeftMouseButton(e) && chosenRelation.get() != null && Main.map.mapView.getEditLayer() != null ) {
                Main.map.mapView.getEditLayer().data.setSelected(chosenRelation.get());
            }
        }

        @Override
        public void mousePressed( MouseEvent e ) {
            checkPopup(e);
        }

        @Override
        public void mouseReleased( MouseEvent e ) {
            checkPopup(e);
        }

        private void checkPopup( MouseEvent e ) {
            if( e.isPopupTrigger() && chosenRelation.get() != null ) {
                popupMenu.show(e.getComponent(), e.getX(), e.getY());
            }
        }
    }

    private class ChosenRelationPopupMenu extends JPopupMenu {
        public ChosenRelationPopupMenu() {
            add(new SelectMembersAction(chosenRelation));
            add(new DeleteChosenRelationAction(chosenRelation));
        }
    }

    private class TogglePanelAction extends AbstractAction {
        private JComponent component;

        public TogglePanelAction( JPanel panel ) {
            super("R");
            this.component = panel;
        }

        public void actionPerformed( ActionEvent e ) {
            component.setVisible(!component.isVisible());
        }
    }

    protected void applyRoleToSelection( String role ) {
        if( chosenRelation != null && chosenRelation.get() != null && Main.main.getCurrentDataSet() != null && !Main.main.getCurrentDataSet().selectionEmpty() ) {
            Collection<OsmPrimitive> selected = Main.main.getCurrentDataSet().getSelected();
            Relation r = new Relation(chosenRelation.get());
            boolean fixed = false;
            for( int i = 0; i < r.getMembersCount(); i++ ) {
                RelationMember m = r.getMember(i);
                if( selected.contains(m.getMember()) ) {
                    if( !role.equals(m.getRole()) ) {
                        r.setMember(i, new RelationMember(role, m.getMember()));
                        fixed = true;
                    }
                }
            }
            if( fixed )
                Main.main.undoRedo.add(new ChangeCommand(chosenRelation.get(), r));
        }
    }

    private class ApplyNewRoleAction extends AbstractAction {
        public ApplyNewRoleAction() {
            super("OK");
        }

        public void actionPerformed( ActionEvent e ) {
            Object selectedItem = roleBox == null ? null : roleBox.getSelectedItem();
            if( selectedItem != null ) {
                if( selectedItem instanceof AutoCompletionListItem )
                    selectedItem = ((AutoCompletionListItem)selectedItem).getValue();
                applyRoleToSelection(selectedItem.toString().trim());
            }
        }
    }

    private class EnterNewRoleAction extends AbstractAction {
        public EnterNewRoleAction() {
            super("+");
        }

        public void actionPerformed( ActionEvent e ) {
            String role = askForRoleName();
            if( role != null )
                applyRoleToSelection(role);
        }
    }
}
