package org.openstreetmap.josm.plugins.validator.tests;

import static org.openstreetmap.josm.tools.I18n.tr;

import java.awt.geom.Area;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.openstreetmap.josm.Main;
import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.coor.EastNorth;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmUtils;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.gui.progress.ProgressMonitor;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.QuadBuckets;
import org.openstreetmap.josm.plugins.validator.PreferenceEditor;
import org.openstreetmap.josm.plugins.validator.Severity;
import org.openstreetmap.josm.plugins.validator.Test;
import org.openstreetmap.josm.plugins.validator.TestError;

/**
 * Tests if there are segments that crosses in the same layer
 *
 * @author frsantos
 */
public class UnconnectedWays extends Test
{
    protected static int UNCONNECTED_WAYS = 1301;
    protected static final String PREFIX = PreferenceEditor.PREFIX + "." + UnconnectedWays.class.getSimpleName();

    Set<MyWaySegment> ways;
    Set<Node> endnodes; // nodes at end of way
    Set<Node> endnodes_highway; // nodes at end of way
    Set<Node> middlenodes; // nodes in middle of way
    Set<Node> othernodes; // nodes appearing at least twice
    //NodeSearchCache nodecache;
    QuadBuckets<Node> nodecache;
    Area ds_area;
    DataSet ds;

    double mindist;
    double minmiddledist;
    /**
     * Constructor
     */
    public UnconnectedWays()
    {
        super(tr("Unconnected ways."),
              tr("This test checks if a way has an endpoint very near to another way."));
    }

    @Override
    public void startTest(ProgressMonitor monitor)
    {
        super.startTest(monitor);
        ways = new HashSet<MyWaySegment>();
        endnodes = new HashSet<Node>();
        endnodes_highway = new HashSet<Node>();
        middlenodes = new HashSet<Node>();
        othernodes = new HashSet<Node>();
        mindist = Main.pref.getDouble(PREFIX + ".node_way_distance", 10.0)/6378135.0;
        minmiddledist = Main.pref.getDouble(PREFIX + ".way_way_distance", 0.0)/6378135.0;
        this.ds = Main.main.getCurrentDataSet();
        this.ds_area = ds.getDataSourceArea();
        // This is temporary until we get proper
        // reindexing in the dataset code.
        ArrayList<Node> ntmp = new ArrayList<Node>(ds.nodes);
        ds.nodes.clear();
        ds.nodes.addAll(ntmp);
        ArrayList<Way> wtmp = new ArrayList<Way>(ds.ways);
        ds.ways.clear();
        ds.ways.addAll(wtmp);
    }

    @Override
    public void endTest()
    {
        //Area a = Main.ds.getDataSourceArea();
        Map<Node, Way> map = new HashMap<Node, Way>();
        long last = -1;
        for (int iter = 0; iter < 1; iter++) {
        last = System.currentTimeMillis();
        long last_print = -1;
        int nr = 0;
        Collection<MyWaySegment> tmp_ways = ways;
        for(MyWaySegment s : tmp_ways) {
            nr++;
            long now = System.currentTimeMillis();
            if (now - last_print > 200) {
                System.err.println("processing segment nr: " + nr + " of " + ways.size());
                last_print = now;
            }
            for(Node en : s.nearbyNodes(mindist)) {
                if (en == null)
                    continue;
                //if("turning_circle".equals(en.get("highway")) ||
                //        (isexit != null && isexit) || en.get("barrier") != null)
                //    c4++;
                if(!s.highway)
                    continue;
                if (!endnodes_highway.contains(en))
                    continue;
                Boolean isexit = OsmUtils.getOsmBoolean(en.get("noexit"));
                if("turning_circle".equals(en.get("highway")) ||
                  (isexit != null && isexit) || en.get("barrier") != null)
                    continue;
                // There's a small false-positive here.  Imagine an intersection
                // like a 't'.  If the top part of the 't' is short enough, it
                // will trigger the node at the very top of the 't' to be unconnected
                // to the way that "crosses" the 't'.  We should probably check that
                // the ways to which 'en' belongs are not connected to 's.w'.
                map.put(en, s.w);
            }
        }
        System.out.println("p1 elapesd: " + (System.currentTimeMillis()-last));
        last = System.currentTimeMillis();
        }
        for(Map.Entry<Node, Way> error : map.entrySet())
        {
            errors.add(new TestError(this, Severity.WARNING,
            tr("Way end node near other highway"), UNCONNECTED_WAYS,
            Arrays.asList(error.getKey(), error.getValue())));
        }
        map.clear();
        for(MyWaySegment s : ways)
        {
            for(Node en : s.nearbyNodes(mindist))
            {
                if (endnodes_highway.contains(en) && !s.highway && !s.isArea()) {
                    map.put(en, s.w);
                } else if (endnodes.contains(en) && !s.isArea()) {
                    map.put(en, s.w);
                }
            }
        }
        System.out.println("p2 elapesd: " + (System.currentTimeMillis()-last));
        last = System.currentTimeMillis();
        for(Map.Entry<Node, Way> error : map.entrySet())
        {
            errors.add(new TestError(this, Severity.WARNING,
            tr("Way end node near other way"), UNCONNECTED_WAYS,
            Arrays.asList(error.getKey(), error.getValue())));
        }
        /* the following two use a shorter distance */
        if(minmiddledist > 0.0)
        {
            map.clear();
            for(MyWaySegment s : ways)
            {
                for(Node en : s.nearbyNodes(minmiddledist))
                {
                    if (!middlenodes.contains(en))
                        continue;
                    map.put(en, s.w);
                }
            }
            System.out.println("p3 elapesd: " + (System.currentTimeMillis()-last));
            last = System.currentTimeMillis();
            for(Map.Entry<Node, Way> error : map.entrySet())
            {
                errors.add(new TestError(this, Severity.OTHER,
                tr("Way node near other way"), UNCONNECTED_WAYS,
                Arrays.asList(error.getKey(), error.getValue())));
            }
            map.clear();
            for(MyWaySegment s : ways)
            {
                for(Node en : s.nearbyNodes(minmiddledist))
                {
                    if (!othernodes.contains(en))
                        continue;
                    map.put(en, s.w);
                }
            }
            System.out.println("p4 elapesd: " + (System.currentTimeMillis()-last));
            last = System.currentTimeMillis();
            for(Map.Entry<Node, Way> error : map.entrySet())
            {
                errors.add(new TestError(this, Severity.OTHER,
                tr("Connected way end node near other way"), UNCONNECTED_WAYS,
                Arrays.asList(error.getKey(), error.getValue())));
            }
        }
        ways = null;
        endnodes = null;
        super.endTest();
        System.out.println("p99 elapesd: " + (System.currentTimeMillis()-last));
        last = System.currentTimeMillis();
    }

    private class MyWaySegment
    {
        private Line2D line;
        public Way w;
        public boolean isAbandoned = false;
        public boolean isBoundary = false;
        public boolean highway;
        private double len;
        private Set<Node> nearbyNodeCache;
        double nearbyNodeCacheDist = -1.0;
        Node n1;
        Node n2;

        public MyWaySegment(Way w, Node n1, Node n2)
        {
            this.w = w;
            String railway = w.get("railway");
            String highway = w.get("highway");
            this.isAbandoned = "abandoned".equals(railway) || "yes".equals(w.get("disused"));
            this.highway = (highway != null || railway != null) && !isAbandoned;
            this.isBoundary = !this.highway && "administrative".equals(w.get("boundary"));
            line = new Line2D.Double(n1.getEastNorth().east(), n1.getEastNorth().north(),
                                     n2.getEastNorth().east(), n2.getEastNorth().north());
            len = line.getP1().distance(line.getP2());
            this.n1 = n1;
            this.n2 = n2;
        }

        public boolean nearby(Node n, double dist)
        {
//            return !w.containsNode(n)
//            && line.ptSegDist(n.getEastNorth().east(), n.getEastNorth().north()) < dist;
            if (w == null)
                Main.debug("way null");
            if (w.containsNode(n))
                return false;
            EastNorth coord = n.getEastNorth();
            if (coord == null)
                return false;
            Point2D p = new Point2D.Double(coord.east(), coord.north());
            if (line.getP1().distance(p) > len+dist)
                return false;
            if (line.getP2().distance(p) > len+dist)
                return false;
            return line.ptSegDist(p) < dist;
        }
        public List<LatLon> getBounds(double fudge)
        {
            double x1 = n1.getCoor().lon();
            double x2 = n2.getCoor().lon();
            if (x1 > x2) {
                double tmpx = x1;
                x1 = x2;
                x2 = tmpx;
            }
            double y1 = n1.getCoor().lat();
            double y2 = n2.getCoor().lat();
            if (y1 > y2) {
                double tmpy = y1;
                y1 = y2;
                y2 = tmpy;
            }
            LatLon topLeft  = new LatLon(y2+fudge, x1-fudge);
            LatLon botRight = new LatLon(y1-fudge, x2+fudge);
            List<LatLon> ret = new ArrayList<LatLon>();
            ret.add(topLeft);
            ret.add(botRight);
            return ret;
        }

        public Collection<Node> nearbyNodes(double dist)
        {
            // If you're looking for nodes that are farther
            // away that we looked for last time, the cached
            // result is no good
            if (dist > nearbyNodeCacheDist) {
                //if (nearbyNodeCacheDist != -1)
                //    System.out.println("destroyed MyWaySegment nearby node cache:" + dist + " > " +  nearbyNodeCacheDist);
                nearbyNodeCache = null;
            }
            if (nearbyNodeCache != null) {
                // If we've cached an aread greater than the
                // one now being asked for...
                if (nearbyNodeCacheDist > dist) {
                    System.out.println("had to trim MyWaySegment nearby node cache.");
                    // Used the cached result and trim out
                    // the nodes that are not in the smaller
                    // area, but keep the old larger cache.
                    Set<Node> trimmed = new HashSet<Node>(nearbyNodeCache);
                    for (Node n : new HashSet<Node>(nearbyNodeCache)) {
                        if (!nearby(n, dist))
                            trimmed.remove(n);
                    }
                    return trimmed;
                }
                return nearbyNodeCache;
            }
            /*
             * We know that any point near the line must be at
             * least as close as the other end of the line, plus
             * a little fudge for the distance away ('dist').
             */

            // This needs to be a hash set because the searches
            // overlap a bit and can return duplicate nodes.
            nearbyNodeCache = null;
            nodecache = ds.nodes;
            List<LatLon> bounds = this.getBounds(dist);
            List<Node> found_nodes = nodecache.search(bounds.get(0), bounds.get(1));
            if (found_nodes == null)
                return Collections.emptySet();

            for (Node n : found_nodes) {
                if (!nearby(n, dist) ||
                     (ds_area != null && !ds_area.contains(n.getCoor())))
                    continue;
                // It is actually very rare for us to find a node
                // so defer as much of the work as possible, like
                // allocating the hash set
                if (nearbyNodeCache == null)
                    nearbyNodeCache = new HashSet<Node>();
                nearbyNodeCache.add(n);
            }
            nearbyNodeCacheDist = dist;
            if (nearbyNodeCache == null)
                nearbyNodeCache = Collections.emptySet();
            return nearbyNodeCache;
        }

        public boolean isArea() {
            return w.get("landuse") != null
                || w.get("leisure") != null
                || w.get("building") != null;
        }
    }

    List<MyWaySegment> getWaySegments(Way w)
    {
        List<MyWaySegment> ret = new ArrayList<MyWaySegment>();
        if (!w.isUsable()
            || w.get("barrier") != null
            || "cliff".equals(w.get("natural")))
            return ret;

        int size = w.getNodesCount();
        if(size < 2)
            return ret;
        for(int i = 1; i < size; ++i)
        {
            if(i < size-1)
                addNode(w.getNode(i), middlenodes);
            MyWaySegment ws = new MyWaySegment(w, w.getNode(i-1), w.getNode(i));
            if (ws.isBoundary || ws.isAbandoned)
                continue;
            ret.add(ws);
        }
        return ret;
    }

    @Override
    public void visit(Way w)
    {
        ways.addAll(getWaySegments(w));
        Set<Node> set = endnodes;
        if(w.get("highway") != null || w.get("railway") != null)
            set = endnodes_highway;
        addNode(w.firstNode(), set);
        addNode(w.lastNode(), set);
    }
    @Override
    public void visit(Node n)
    {
    }
    private void addNode(Node n, Set<Node> s)
    {
        Boolean m = middlenodes.contains(n);
        Boolean e = endnodes.contains(n);
        Boolean eh = endnodes_highway.contains(n);
        Boolean o = othernodes.contains(n);
        if(!m && !e && !o && !eh)
            s.add(n);
        else if(!o)
        {
            othernodes.add(n);
            if(e)
                endnodes.remove(n);
            else if(eh)
                endnodes_highway.remove(n);
            else
                middlenodes.remove(n);
        }
    }
}
