// License: GPL. Copyright 2007 by Immanuel Scholz and others
package org.openstreetmap.josm.actions.downloadtasks;

import static org.openstreetmap.josm.tools.I18n.tr;

import java.io.IOException;
import java.util.concurrent.Future;

import javax.swing.JCheckBox;

import org.openstreetmap.josm.Main;
import org.openstreetmap.josm.actions.DownloadAction;
import org.openstreetmap.josm.data.Bounds;
import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.DataSource;
import org.openstreetmap.josm.gui.PleaseWaitRunnable;
import org.openstreetmap.josm.gui.download.DownloadDialog.DownloadTask;
import org.openstreetmap.josm.gui.layer.OsmDataLayer;
import org.openstreetmap.josm.gui.progress.NullProgressMonitor;
import org.openstreetmap.josm.gui.progress.ProgressMonitor;
import org.openstreetmap.josm.io.BoundingBoxDownloader;
import org.openstreetmap.josm.io.OsmServerLocationReader;
import org.openstreetmap.josm.io.OsmServerReader;
import org.openstreetmap.josm.io.OsmTransferException;
import org.xml.sax.SAXException;


/**
 * Open the download dialog and download the data.
 * Run in the worker thread.
 */
public class DownloadOsmTask implements DownloadTask {
    private static Bounds currentBounds;
    private Future<Task> task = null;
    private DataSet downloadedData;

    private class Task extends PleaseWaitRunnable {
        private OsmServerReader reader;
        private DataSet dataSet;
        private boolean newLayer;

        public Task(boolean newLayer, OsmServerReader reader, ProgressMonitor progressMonitor) {
            super(tr("Downloading data"), progressMonitor, false);
            this.reader = reader;
            this.newLayer = newLayer;
        }

        @Override public void realRun() throws IOException, SAXException, OsmTransferException {
            dataSet = reader.parseOsm(progressMonitor.createSubTaskMonitor(ProgressMonitor.ALL_TICKS, false));
        }

        protected OsmDataLayer getEditLayer() {
            if (Main.map == null) return null;
            if (Main.map.mapView == null) return null;
            return Main.map.mapView.getEditLayer();
        }

        @Override protected void finish() {
            if (dataSet == null)
                return; // user canceled download or error occurred
            if (dataSet.allPrimitives().isEmpty()) {
                progressMonitor.setErrorMessage(tr("No data imported."));
                // need to synthesize a download bounds lest the visual indication of downloaded
                // area doesn't work
                dataSet.dataSources.add(new DataSource(currentBounds, "OpenStreetMap server"));
            }
            rememberDownloadedData(dataSet);
            if (newLayer || getEditLayer() == null) {
                OsmDataLayer layer = new OsmDataLayer(dataSet, OsmDataLayer.createNewName(), null);
                Main.main.addLayer(layer);
            } else {
                getEditLayer().mergeFrom(dataSet);
            }
        }

        @Override protected void cancel() {
            if (reader != null) {
                reader.cancel();
            }
        }
    }
    private JCheckBox checkBox = new JCheckBox(tr("OpenStreetMap data"), true);

    private void rememberDownloadedData(DataSet ds) {
        this.downloadedData = ds;
    }

    public DataSet getDownloadedData() {
        return downloadedData;
    }

    public void download(DownloadAction action, double minlat, double minlon,
            double maxlat, double maxlon, ProgressMonitor progressMonitor) {
        // Swap min and max if user has specified them the wrong way round
        // (easy to do if you are crossing 0, for example)
        // FIXME should perhaps be done in download dialog?
        if (minlat > maxlat) {
            double t = minlat; minlat = maxlat; maxlat = t;
        }
        if (minlon > maxlon) {
            double t = minlon; minlon = maxlon; maxlon = t;
        }

        boolean newLayer = action != null
        && (action.dialog == null || action.dialog.newLayer.isSelected());

        Task t = new Task(newLayer,
                new BoundingBoxDownloader(minlat, minlon, maxlat, maxlon), progressMonitor);
        currentBounds = new Bounds(new LatLon(minlat, minlon), new LatLon(maxlat, maxlon));
        // We need submit instead of execute so we can wait for it to finish and get the error
        // message if necessary. If no one calls getErrorMessage() it just behaves like execute.
        task = Main.worker.submit(t, t);
    }

    /**
     * Loads a given URL from the OSM Server
     * @param True if the data should be saved to a new layer
     * @param The URL as String
     */
    public void loadUrl(boolean new_layer, String url) {
        Task t = new Task(new_layer,
                new OsmServerLocationReader(url),
                NullProgressMonitor.INSTANCE);
        task = Main.worker.submit(t, t);
    }

    public JCheckBox getCheckBox() {
        return checkBox;
    }

    public String getPreferencesSuffix() {
        return "osm";
    }

    /*
     * (non-Javadoc)
     * @see org.openstreetmap.josm.gui.download.DownloadDialog.DownloadTask#getErrorMessage()
     */
    public String getErrorMessage() {
        if(task == null)
            return "";

        try {
            Task t = task.get();
            return t.getProgressMonitor().getErrorMessage() == null
            ? ""
                    : t.getProgressMonitor().getErrorMessage();
        } catch (Exception e) {
            return "";
        }
    }
}
